package ru.yandex.travel.api.services.orders;

import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.models.train.ApiErrorCode;
import ru.yandex.travel.api.models.train.ApiErrorMessageCode;
import ru.yandex.travel.api.models.train.ErrorInfo;
import ru.yandex.travel.api.models.train.ErrorType;
import ru.yandex.travel.api.models.train.TrainOrderStatus;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.train.model.ErrorMessageCode;
import ru.yandex.travel.workflow.EWorkflowState;

// todo(tlg-13,mbobrov): the whole class should be removed, use the status from the orders app instead
@Service
@Slf4j
public class TrainOrderStatusMappingService {
    public TrainOrderStatus getStatus(TOrderInfo order) {
        Preconditions.checkArgument(order.getOrderType() == EOrderType.OT_GENERIC,
                "Unsupported order type: type %s, order id %s", order.getOrderType(), order.getOrderId());
        ETrainOrderState orderState = order.getTrainOrderState();
        EWorkflowState workflowState = order.getWorkflowState();
        ETrustInvoiceState currentInvoiceState = ETrustInvoiceState.IS_UNKNOWN;
        if (order.hasCurrentInvoice()) {
            currentInvoiceState = order.getCurrentInvoice().getTrustInvoiceState();
        }
        var isOrderExpired = false;
        if (order.hasExpiresAt()) {
            isOrderExpired = ProtoUtils.toInstant(order.getExpiresAt()).isBefore(Instant.now());
        }
        return getStatusGeneric(order.getGenericOrderState(), workflowState,
                currentInvoiceState, order.getUserActionScheduled(), isOrderExpired);
    }

    private TrainOrderStatus getStatusGeneric(EOrderState orderState, EWorkflowState orderWorkflowState,
                                              ETrustInvoiceState currentInvoiceState,
                                              boolean userActionScheduled, boolean isOrderExpired) {
        if (orderWorkflowState == EWorkflowState.WS_CRASHED) {
            //noinspection SwitchStatementWithTooFewBranches
            switch (orderState) {
                case OS_WAITING_CONFIRMATION:
                    return TrainOrderStatus.CONFIRMATION_FAILED;
            }
            // TODO (mbobrov, ganintsev): think of a failed refund
            return TrainOrderStatus.CANCELLED;
        }
        switch (orderState) {
            case OS_CANCELLED:
            case OS_WAITING_CANCELLATION:
                return TrainOrderStatus.CANCELLED;
            case OS_RESERVED:
            case OS_WAITING_PAYMENT:
                if (currentInvoiceState == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                    if (isOrderExpired) {
                        return TrainOrderStatus.CANCELLED;
                    }
                    return TrainOrderStatus.PAYMENT_FAILED;
                } else {
                    return TrainOrderStatus.WAITING_PAYMENT;
                }
            case OS_PAYMENT_FAILED:
                return TrainOrderStatus.PAYMENT_FAILED;
            case OS_WAITING_CONFIRMATION:
                return TrainOrderStatus.WAITING_CONFIRMATION;
            case OS_CONFIRMED:
                if (userActionScheduled) {
                    return TrainOrderStatus.IN_PROGRESS;
                } else {
                    return TrainOrderStatus.CONFIRMED;
                }
            case OS_REFUNDED:
                return TrainOrderStatus.CONFIRMED;
            case OS_REFUNDING:
                return TrainOrderStatus.WAITING_REFUND;
            case OS_NEW:
            case OS_WAITING_RESERVATION:
                return TrainOrderStatus.WAITING_RESERVATION;
            default:
                log.warn("Unknown order state {}", orderState);
        }
        return TrainOrderStatus.CANCELLED;
    }

    public ErrorInfo getErrorInfo(TOrderInfo order, ru.yandex.travel.train.model.ErrorInfo sourceErrorInfo) {
        if (order != null) {
            EWorkflowState workflowState = order.getWorkflowState();
            if (workflowState == EWorkflowState.WS_CRASHED) {
                var result = new ErrorInfo();
                result.setType(ErrorType.PROCESS_EXCEPTION_STATE);
                return result;
            }
        }
        if (sourceErrorInfo != null) {
            var result = new ErrorInfo();
            result.setCode(ApiErrorCode.BY_MODEL_VALUE.getByValueOrNull(sourceErrorInfo.getCode()));
            result.setMessageCode(ApiErrorMessageCode.BY_MODEL_VALUE.getByValueOrNull(
                    ErrorMessageCode.BY_ERROR_CODE.getByValueOrDefault(sourceErrorInfo.getCode(), ErrorMessageCode.PARTNER_ERROR)
            ));
            result.setMessage(sourceErrorInfo.getMessage());
            result.setMessageParams(sourceErrorInfo.getMessageParams());
            result.setType(ErrorType.fromErrorMessageCode(sourceErrorInfo.getCode()));
            return result;
        }
        return null;
    }
}
