package ru.yandex.travel.api.services.orders;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import io.opentracing.Span;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.AddInsuranceReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CalculateRefundAmountReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CalculateRefundAmountRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CancelOrderReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.ChangeRegistrationStatusReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CreateOrderReqV2;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.CreateOrderRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.DownloadBlankReqV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.OrderInfoRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.OrderStatusRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.RefundReqV1;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.services.train.TrainOfferService;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReq;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.workflow.train.proto.TTrainCalculateRefundReqInfo;
import ru.yandex.travel.tracing.TracerHelpers;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;

@Service
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties(TrainOrdersServiceProperties.class)
public class TrainOrdersService {

    private final OrchestratorClientFactory orchestratorClientFactory;
    private final TrainModelMapService trainModelMapService;
    private final TrainReservationMapService trainReservationMapService;
    private final Meters meters;
    private final ImClient imClient;
    private final Tracer tracer;
    private final Retry retryHelper;
    private final TrainOfferService offerService;
    private final TrainOrdersServiceProperties trainOrdersServiceProperties;
    private final ApiTokenEncrypter apiTokenEncrypter;

    public CompletableFuture<CreateOrderRspV1> createOrder(CreateOrderReqV2 request) {
        try {
            return createOrderInner(request);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private CompletableFuture<CreateOrderRspV1> createOrderInner(CreateOrderReqV2 request) {
        Span span = tracer.activeSpan();
        // todo(ganintsev): the whole class should be removed in TRAVELBACK-1828
        throw new UnsupportedOperationException("Legacy train orders are disabled");
    }

    public CompletableFuture<Void> payOrder(String orderId, String returnUrl, String customerSource, boolean newCommonPaymentWebForm) {
        if (Strings.isNullOrEmpty(customerSource)) {
            customerSource = "desktop";
        }
        final String finalCustomerSource = customerSource;

        try (var ignoredMdc = NestedMdc.forEntityId(orderId)) {
            log.info("Starting payment with url \"{}\" and source \"{}\"", returnUrl, customerSource);
            var future = withClient().startPayment(
                    TStartPaymentReq.newBuilder()
                            .setCallId(CallIdHelper.callIdForMethod("startPayment"))
                            .setInvoiceType(EInvoiceType.IT_TRUST)
                            .setOrderId(orderId)
                            .setReturnUrl(returnUrl)
                            .setSource(finalCustomerSource)
                            .setNewCommonPaymentWebForm(newCommonPaymentWebForm)
                            .build()
            );
            return FutureUtils.buildCompletableFuture(future).thenApply(ignored -> null);
        }
    }

    public CompletableFuture<OrderStatusRspV1> getOrderStatus(String id) {
        if (trainOrdersServiceProperties.isAggregateStatusEnabled()) {
            return FutureUtils.buildCompletableFuture(
                    withSlavePreferred().getOrderAggregateState(TGetOrderAggregateStateReq.newBuilder().setOrderId(id).build())
            ).thenApply(aggState -> {
                var result = new OrderStatusRspV1();
                trainModelMapService.fillStatusResponseFromAggregate(result, aggState.getOrderAggregateState());
                return result;
            });
        } else {
            return FutureUtils.buildCompletableFuture(
                    withSlavePreferred().getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(id).build())
            ).thenApply(orderRsp -> {
                var result = new OrderStatusRspV1();
                trainModelMapService.fillStatusResponse(result, orderRsp.getResult());
                return result;
            });
        }
    }

    public CompletableFuture<byte[]> downloadBlank(DownloadBlankReqV1 request) {
        Span span = tracer.activeSpan();
        UUID orderId = request.getId();
        try (var ignoredMdc = NestedMdc.forEntityId(orderId)) {
            log.info(String.format("Downloading blank, user %s", UserCredentials.get().getLogin()));
            return FutureUtils.buildCompletableFuture(
                            withClient().getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId.toString()).build()))
                    .thenApply(orderInfo -> createGetOrderReservationBlankRequest(orderInfo))
                    .thenCompose(FutureUtils.withMdc(TracerHelpers.applyWithActiveSpanScope(
                                    tracer, span, blankRequest ->
                                            imClient.orderReservationBlankAsync(
                                                    blankRequest,
                                                    trainOrdersServiceProperties.getDownloadBlankDuration()
                                ))
                    ));
        }
    }


    public CompletableFuture<OrderInfoRspV1> getOrderInfo(String id, boolean updateOrderOnTheFly) {
        try (var ignoredMdc = NestedMdc.forEntityId(id)) {
            log.info("Getting order info with update on the fly: {}", updateOrderOnTheFly);
            var client = withClient();
            if (updateOrderOnTheFly) {
                // TODO (mbobrov): think of propagating time remaining through retries
                client = client.withDeadlineAfter(
                        trainOrdersServiceProperties.getGetActualizedOrderInfoDuration().toMillis(),
                        TimeUnit.MILLISECONDS
                );
            }
            return FutureUtils.buildCompletableFuture(
                    client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(id)
                            .setUpdateOrderOnTheFly(updateOrderOnTheFly).build()
                    )
            ).thenApply(orderRsp -> {
                TOrderInfo orderInfo = orderRsp.getResult();
                return trainModelMapService.convert(orderInfo, false);
            });
        }
    }

    public CompletableFuture<Void> addInsurance(AddInsuranceReqV1 request) {
        return FutureUtils.buildCompletableFuture(withClient().addInsurance(TAddInsuranceReq.newBuilder()
                .setCallId(CallIdHelper.callIdForMethod("addInsurance"))
                .setOrderId(request.getId()).build())).thenApply(ignored -> null);
    }

    public CompletableFuture<Void> cancelOrder(CancelOrderReqV1 request) {
        return FutureUtils.buildCompletableFuture(
                withClient().startCancellation(
                        TStartCancellationReq.newBuilder().setCallId(CallIdHelper.callIdForMethod("cancelOrder"))
                                .setOrderId(request.getId()).build())
        ).thenApply(ignored -> null);
    }

    public CompletableFuture<CalculateRefundAmountRspV1> calculateRefundAmount(CalculateRefundAmountReqV1 request) {
        var req = TCalculateRefundReq.newBuilder().setOrderId(request.getOrderId());
        var trainReq = TTrainCalculateRefundReqInfo.newBuilder()
                .addAllBlankIds(request.getBlankIds());
        req.setTrainCalculateRefundInfo(trainReq.build());
        return FutureUtils.buildCompletableFuture(withClient().calculateRefund(req.build()))
                .thenApply(calculateRefundRsp -> {
                    var result = new CalculateRefundAmountRspV1();
                    result.setExpiresAt(ProtoUtils.toInstant(calculateRefundRsp.getExpiresAt()));
                    result.setRefundToken(calculateRefundRsp.getRefundToken());
                    result.setRefundAmount(ProtoUtils.fromTPrice(calculateRefundRsp.getRefundAmount()));
                    return result;
                });

    }


    public CompletableFuture<Void> startRefund(RefundReqV1 request) {
        return FutureUtils.buildCompletableFuture(
                        withClient().startRefund(TStartRefundReq.newBuilder()
                                .setCallId(CallIdHelper.callIdForMethod("startRefund"))
                                .setOrderId(request.getOrderId())
                                .setRefundToken(request.getRefundToken()).build()))
                .thenApply(ignored -> {
                    meters.getTrainOrdersRefunded().increment();
                    return null;
                });
    }

    public CompletableFuture<Void> changeRegistrationStatus(ChangeRegistrationStatusReqV1 request) {
        String orderId = request.getOrderId();
        String blankIds = request.getBlankIds().stream()
                .map(String::valueOf).collect(Collectors.joining(", "));
        try (var ignoredMdc = NestedMdc.forEntityId(orderId)) {
            log.info("Changing registration status for blanks [{}], enabled: {}", blankIds, request.getEnabled());
            return FutureUtils.buildCompletableFuture(
                            withClient().changeTrainRegistrationStatus(TChangeTrainRegistrationStatusReq.newBuilder()
                                    .setCallId(CallIdHelper.callIdForMethod("changeRegistration"))
                                    .setOrderId(orderId)
                                    .setEnabled(request.getEnabled())
                                    .addAllBlankIds(request.getBlankIds()).build()))
                    .thenApply(ignored -> null);
        }
    }

    private OrderReservationBlankRequest createGetOrderReservationBlankRequest(TGetOrderInfoRsp orderInfo) {
        TJson payloadJson = orderInfo.getResult().getService(0).getServiceInfo().getPayload();
        TrainReservation payload = ProtoUtils.fromTJson(payloadJson, TrainReservation.class);
        return new OrderReservationBlankRequest(payload.getPartnerOrderId(), null);
    }

    private CompletableFuture<CreateOrderRspV1> createAndReserveOrder(TCreateOrderReq req, String orderHistory,
                                                                      Span span) {
        AtomicReference<String> createdOrderId = new AtomicReference<>();
        return FutureUtils.buildCompletableFuture(
                        withClient().createOrder(req))
                .thenCompose(TracerHelpers.applyWithActiveSpanScope(tracer, span,
                        createResp -> {
                            createdOrderId.set(createResp.getNewOrder().getOrderId());
                            return FutureUtils.buildCompletableFuture(
                                    withClient().reserve(TReserveReq.newBuilder()
                                            .setOrderId(createResp.getNewOrder().getOrderId())
                                            .setCallId(CallIdHelper.callIdForMethod("reserveOrder"))
                                            .build()
                                    ));
                        }))
                .thenApply(orderId -> {
                    try (var ignoredMdc = NestedMdc.forEntityId(createdOrderId.get())) {
                        log.info("Created order");
                        log.info(orderHistory);
                    }
                    meters.getTrainOrdersCreated().increment();
                    CreateOrderRspV1 order = new CreateOrderRspV1();
                    order.setId(createdOrderId.get());
                    return order;
                });
    }

    private OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub withClient() {
        return orchestratorClientFactory.createFutureStubForTrains();
    }

    private OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub withSlavePreferred() {
        return orchestratorClientFactory.createFutureStubForTrainsReadOnly();
    }
}
