package ru.yandex.travel.api.services.orders;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import io.sentry.Sentry;
import io.sentry.event.Event;
import io.sentry.event.EventBuilder;
import io.sentry.event.interfaces.ExceptionInterface;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.generic_booking_flow.model.CreateTrainServiceData;
import ru.yandex.travel.api.models.train.CreateOrderPassengerV2;
import ru.yandex.travel.api.services.dictionaries.country.CountryDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.readable_timezone.TrainReadableTimezoneDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station.TrainStationDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station_code.TrainStationCodeDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station_express_alias.TrainStationExpressAliasDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.time_zone.TrainTimeZoneDataProvider;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;
import ru.yandex.travel.dicts.rasp.proto.TStation;
import ru.yandex.travel.train.model.AdditionalPlaceRequirements;
import ru.yandex.travel.train.model.CabinGenderKind;
import ru.yandex.travel.train.model.CabinPlaceDemands;
import ru.yandex.travel.train.model.CarStorey;
import ru.yandex.travel.train.model.CarType;
import ru.yandex.travel.train.model.Direction;
import ru.yandex.travel.train.model.DocumentType;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.RailwayBonusCard;
import ru.yandex.travel.train.model.RoutePolicy;
import ru.yandex.travel.train.model.Sex;
import ru.yandex.travel.train.model.TrainDicts;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainReservationRequestData;
import ru.yandex.travel.train.model.TrainReservationUiData;
import ru.yandex.travel.trains.proto.EAdditionalPlaceRequirements;
import ru.yandex.travel.trains.proto.ECabinGenderKind;
import ru.yandex.travel.trains.proto.ECabinPlaceDemands;
import ru.yandex.travel.trains.proto.ECarStorey;
import ru.yandex.travel.trains.proto.ECarType;
import ru.yandex.travel.trains.proto.EDirection;
import ru.yandex.travel.trains.proto.EDocumentType;
import ru.yandex.travel.trains.proto.ELoyaltyCardType;
import ru.yandex.travel.trains.proto.EPassengerCategory;
import ru.yandex.travel.trains.proto.ERoutePolicy;
import ru.yandex.travel.trains.proto.ESex;
import ru.yandex.travel.trains.proto.TLoyaltyCard;
import ru.yandex.travel.trains.proto.TPassenger;
import ru.yandex.travel.trains.proto.TTrainServiceOffer;

@Service
@Slf4j
@RequiredArgsConstructor
public class TrainReservationMapService {
    private final CountryDataProvider countryDataProvider;
    private final TrainReadableTimezoneDataProvider trainReadableTimezoneDataProvider;
    private final TrainSettlementDataProvider trainSettlementDataProvider;
    private final TrainStationDataProvider trainStationDataProvider;
    private final TrainStationCodeDataProvider trainStationCodeDataProvider;
    private final TrainTimeZoneDataProvider trainTimeZoneDataProvider;
    private final TrainStationExpressAliasDataProvider trainStationExpressAliasDataProvider;

    public static final String DASH = "\u2014";
    private static final Integer MOSCOW_REGION_ID = 1;
    private static final Integer MOSCOW_REGION_GEO_ID = 1;
    private static final String DEFAULT_LANGUAGE = "ru";

    private static final Map<EAdditionalPlaceRequirements, AdditionalPlaceRequirements> ADDITIONAL_PLACE_REQUIREMENTS_MAP = Map.ofEntries(
            Map.entry(EAdditionalPlaceRequirements.PR_UNKNOWN, AdditionalPlaceRequirements.NO_VALUE),
            Map.entry(EAdditionalPlaceRequirements.PR_MOTHER_AND_BABY_PLACES,
                    AdditionalPlaceRequirements.MOTHER_AND_BABY_PLACES),
            Map.entry(EAdditionalPlaceRequirements.PR_WITH_BABY_PLACES, AdditionalPlaceRequirements.WITH_BABY_PLACES),
            Map.entry(EAdditionalPlaceRequirements.PR_WITH_PETS_PLACES, AdditionalPlaceRequirements.WITH_PETS_PLACES),
            Map.entry(EAdditionalPlaceRequirements.PR_USUAL, AdditionalPlaceRequirements.USUAL),
            Map.entry(EAdditionalPlaceRequirements.PR_USUAL_NEAR_THE_TABLE,
                    AdditionalPlaceRequirements.USUAL_NEAR_THE_TABLE),
            Map.entry(EAdditionalPlaceRequirements.PR_ANY_NEAR_THE_TABLE,
                    AdditionalPlaceRequirements.ANY_NEAR_THE_TABLE),
            Map.entry(EAdditionalPlaceRequirements.PR_ANY_NOT_NEAR_THE_TABLE,
                    AdditionalPlaceRequirements.ANY_NOT_NEAR_THE_TABLE),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_THE_PLAYGROUND,
                    AdditionalPlaceRequirements.NEAR_THE_PLAYGROUND),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_THE_PLAYGROUND_AND_NOT_THE_TABLE,
                    AdditionalPlaceRequirements.NEAR_THE_PLAYGROUND_AND_NOT_THE_TABLE),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_THE_PLAYGROUND_AND_THE_TABLE,
                    AdditionalPlaceRequirements.NEAR_THE_PLAYGROUND_AND_THE_TABLE),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_THE_PLACES_WITH_PETS,
                    AdditionalPlaceRequirements.NEAR_THE_PLACES_WITH_PETS),
            Map.entry(EAdditionalPlaceRequirements.PR_FOLDABLE_PLACE, AdditionalPlaceRequirements.FOLDABLE_PLACE),
            Map.entry(EAdditionalPlaceRequirements.PR_FORWARD, AdditionalPlaceRequirements.FORWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_BACKWARD, AdditionalPlaceRequirements.BACKWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_WINDOW, AdditionalPlaceRequirements.NEAR_WINDOW),
            Map.entry(EAdditionalPlaceRequirements.PR_UNFOLDABLE_PLACE, AdditionalPlaceRequirements.UNFOLDABLE_PLACE),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_THE_TABLE_AND_BACKWARD,
                    AdditionalPlaceRequirements.NEAR_THE_TABLE_AND_BACKWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_THE_TABLE_AND_FORWARD,
                    AdditionalPlaceRequirements.NEAR_THE_TABLE_AND_FORWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_WITHOUT_TABLE_AND_BACKWARD,
                    AdditionalPlaceRequirements.WITHOUT_TABLE_AND_BACKWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_WITHOUT_TABLE_AND_FORWARD,
                    AdditionalPlaceRequirements.WITHOUT_TABLE_AND_FORWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_WITHOUT_WINDOW_AND_BACKWARD,
                    AdditionalPlaceRequirements.WITHOUT_WINDOW_AND_BACKWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_WITHOUT_WINDOW_AND_FORWARD,
                    AdditionalPlaceRequirements.WITHOUT_WINDOW_AND_FORWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_SINGLE_AND_FORWARD,
                    AdditionalPlaceRequirements.SINGLE_AND_FORWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_RESTROOM, AdditionalPlaceRequirements.NEAR_RESTROOM),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_RESTROOM_AND_BACKWARD,
                    AdditionalPlaceRequirements.NEAR_RESTROOM_AND_BACKWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_NEAR_RESTROOM_AND_FORWARD,
                    AdditionalPlaceRequirements.NEAR_RESTROOM_AND_FORWARD),
            Map.entry(EAdditionalPlaceRequirements.PR_NO_TABLE_AND_NO_WINDOW,
                    AdditionalPlaceRequirements.NO_TABLE_AND_NO_WINDOW)
    );
    private static final Map<ECabinGenderKind, CabinGenderKind> CABIN_GENDER_KIND_MAP = Map.of(
            ECabinGenderKind.GK_UNKNOWN, CabinGenderKind.NO_VALUE,
            ECabinGenderKind.GK_MIXED, CabinGenderKind.MIXED,
            ECabinGenderKind.GK_MALE, CabinGenderKind.MALE,
            ECabinGenderKind.GK_FEMALE, CabinGenderKind.FEMALE
    );
    private static final Map<ECarType, CarType> CAR_TYPE_MAP = Map.of(
            ECarType.CT_UNKNOWN, CarType.UNKNOWN,
            ECarType.CT_SHARED, CarType.SHARED,
            ECarType.CT_SOFT, CarType.SOFT,
            ECarType.CT_LUXURY, CarType.LUXURY,
            ECarType.CT_COMPARTMENT, CarType.COMPARTMENT,
            ECarType.CT_RESERVED_SEAT, CarType.RESERVED_SEAT,
            ECarType.CT_SEDENTARY, CarType.SEDENTARY,
            ECarType.CT_BAGGAGE, CarType.BAGGAGE
    );
    private static final Map<ECabinPlaceDemands, CabinPlaceDemands> CABIN_PLACE_DEMANDS_MAP = Map.of(
            ECabinPlaceDemands.PD_UNKNOWN, CabinPlaceDemands.NO_VALUE,
            ECabinPlaceDemands.PD_IN_ONE_CABIN, CabinPlaceDemands.IN_ONE_CABIN,
            ECabinPlaceDemands.PD_NO_SIDE_PLACES, CabinPlaceDemands.NO_SIDE_PLACES,
            ECabinPlaceDemands.PD_IN_ONE_COMPARTMENT, CabinPlaceDemands.IN_ONE_COMPARTMENT
    );
    private static final Map<ECarStorey, CarStorey> CAR_STOREY_MAP = Map.of(
            ECarStorey.CS_UNKNOWN, CarStorey.NO_VALUE,
            ECarStorey.CS_FIRST, CarStorey.FIRST,
            ECarStorey.CS_SECOND, CarStorey.SECOND
    );
    private static final Map<ERoutePolicy, RoutePolicy> ROUTE_POLICY_MAP = Map.of(
            ERoutePolicy.RP_INTERNAL, RoutePolicy.INTERNAL,
            ERoutePolicy.RP_INTERNATIONAL, RoutePolicy.INTERNATIONAL,
            ERoutePolicy.RP_FINLAND, RoutePolicy.FINLAND
    );
    private static final Map<EDocumentType, DocumentType> DOCUMENT_TYPE_MAP = Map.of(
            EDocumentType.DT_RUSSIAN_PASSPORT, DocumentType.RUSSIAN_PASSPORT,
            EDocumentType.DT_BIRTH_CERTIFICATE, DocumentType.BIRTH_CERTIFICATE,
            EDocumentType.DT_RUSSIAN_INTERNATIONAL_PASSPORT, DocumentType.RUSSIAN_INTERNATIONAL_PASSPORT,
            EDocumentType.DT_FOREIGN_DOCUMENT, DocumentType.FOREIGN_DOCUMENT,
            EDocumentType.DT_SAILOR_PASSPORT, DocumentType.SAILOR_PASSPORT,
            EDocumentType.DT_MILITARY_CARD, DocumentType.MILITARY_CARD
    );
    private static final Map<EPassengerCategory, PassengerCategory> PASSENGER_CATEGORY_MAP = Map.of(
            EPassengerCategory.PC_ADULT, PassengerCategory.ADULT,
            EPassengerCategory.PC_CHILD, PassengerCategory.CHILD,
            EPassengerCategory.PC_BABY, PassengerCategory.BABY
    );
    private static final Map<ESex, Sex> SEX_MAP = Map.of(
            ESex.S_MALE, Sex.MALE,
            ESex.S_FEMALE, Sex.FEMALE
    );
    private static final Map<ELoyaltyCardType, String> LOYALTY_CARD_TYPE_MAP = Map.of(
            ELoyaltyCardType.LCT_RZHD_BONUS, "RzhdBonus",
            ELoyaltyCardType.LCT_UNIVERSAL_RZHD_CARD, "UniversalRzhdCard"
    );
    private static final Map<EDirection, Direction> DIRECTION_MAP = Map.of(
            EDirection.D_FORWARD, Direction.FORWARD,
            EDirection.D_BACKWARD, Direction.BACKWARD
    );

    public TrainReservation createTrainReservation(CreateTrainServiceData request, boolean isMobile, Integer geoId,
                                                   TTrainServiceOffer offer) {
        var payload = new TrainReservation();
        payload.setOfferId(request.getOfferId());
        payload.setBanditContext(offer.getBanditContext());
        payload.setBanditToken(offer.getFeeCalculationToken());
        payload.setDirection(DIRECTION_MAP.getOrDefault(offer.getDirection(), Direction.FORWARD));
        payload.setCorrelationId(request.getCorrelationId());
        payload.setSegmentIndex(offer.getSegmentIndex());
        payload.setCompanyTitle(offer.getCompanyTitle());
        Map<Integer, TPassenger> offerPassengersMap = offer.getPassengersList().stream()
                .collect(Collectors.toMap(TPassenger::getPassengerId, x -> x));
        var indexes =
                request.getPassengers().stream().map(CreateOrderPassengerV2::getIndex).collect(Collectors.toSet());
        Preconditions.checkArgument(indexes.equals(offerPassengersMap.keySet()), "Passengers not match offer");
        payload.setPassengers(request.getPassengers().stream()
                .map(x -> convertToTrainPassenger(x, offerPassengersMap.get(x.getIndex())))
                .collect(Collectors.toList()));
        if (!offer.getGiveChildWithoutPlace() &&
                payload.getPassengers().stream().anyMatch(p -> p.getRequestedPlaces().size() > 0)) {
            long errorPassengersCount = payload.getPassengers().stream()
                    .filter(p -> p.getCategory() != PassengerCategory.BABY)
                    .filter(p -> p.getRequestedPlaces().size() == 0)
                    .count();
            Preconditions.checkArgument(errorPassengersCount == 0, "All passengers should be with requested places");
            errorPassengersCount = payload.getPassengers().stream()
                    .filter(p -> p.getCategory() == PassengerCategory.BABY)
                    .filter(p -> p.getRequestedPlaces().size() > 0)
                    .count();
            Preconditions.checkArgument(errorPassengersCount == 0, "All children without places should be without " +
                    "requested places");
        }
        var reservationData = new TrainReservationRequestData();
        payload.setReservationRequestData(reservationData);
        reservationData.setBanditType(offer.getBanditType());
        reservationData.setElectronicRegistrationEnabled(offer.getElectronicRegistration());
        reservationData.setAdditionalPlaceRequirements(ADDITIONAL_PLACE_REQUIREMENTS_MAP.get(offer.getAdditionalPlaceRequirements()));
        switch (offer.getBeddingChoose()) {
            case B_UNKNOWN:
                reservationData.setBedding(offer.getBedding());
                break;
            case B_ANY:
                reservationData.setBedding(null);
                break;
            case B_WITH:
                reservationData.setBedding(true);
                break;
            case B_WITHOUT:
                reservationData.setBedding(false);
                break;
        }
        reservationData.setCarNumber(offer.getCarNumber());
        reservationData.setCarType(CAR_TYPE_MAP.get(offer.getCarType()));
        reservationData.setCabinGenderKind(CABIN_GENDER_KIND_MAP.get(offer.getCabinGenderKind()));
        if (offer.getPlacesList().size() > 0) {
            Integer placeNumberFrom = offer.getPlacesList().stream().min(Integer::compareTo).get();
            Integer placeNumberTo = offer.getPlacesList().stream().max(Integer::compareTo).get();
            reservationData.setPlaceNumberFrom(placeNumberFrom);
            reservationData.setPlaceNumberTo(placeNumberTo);
        }
        if (offer.hasPlaceRequirements()) {
            if (offer.getPlaceRequirements().getCabinPlaceDemands() != ECabinPlaceDemands.PD_UNKNOWN) {
                reservationData.setCabinPlaceDemands(CABIN_PLACE_DEMANDS_MAP.get(offer.getPlaceRequirements().getCabinPlaceDemands()));
            }
            reservationData.setCarStorey(CAR_STOREY_MAP.get(offer.getPlaceRequirements().getCarStorey()));
            if (offer.getPlaceRequirements().hasCreateOrderPlaceCount()) {
                var countRequirements = offer.getPlaceRequirements().getCreateOrderPlaceCount();
                if (countRequirements.getBottom() + countRequirements.getNearWindow() +
                        countRequirements.getNearPassage() + countRequirements.getUpper() > 0) {
                    reservationData.setLowerPlaceQuantity(countRequirements.getBottom() + countRequirements.getNearWindow());
                    reservationData.setUpperPlaceQuantity(countRequirements.getNearPassage() + countRequirements.getUpper());
                }
            }
        }
        reservationData.setDepartureTime(ProtoUtils.toInstant(offer.getDeparture()));
        reservationData.setArrivalTime(ProtoUtils.toInstantFromSafe(offer.getArrival()));
        reservationData.setGiveChildWithoutPlace(offer.getGiveChildWithoutPlace());
        reservationData.setServiceClass(offer.getServiceClass());
        reservationData.setSchemeId(offer.getSchemeId());
        reservationData.setInternationalServiceClass(offer.getInternationalServiceClass());
        var fromCode = trainStationCodeDataProvider.getStationExpressCode(offer.getStationFromId()).getCode();
        var toCode = trainStationCodeDataProvider.getStationExpressCode(offer.getStationToId()).getCode();
        var from = trainStationDataProvider.getById(offer.getStationFromId());
        var to = trainStationDataProvider.getById(offer.getStationToId());

        reservationData.setStationFromId(offer.getStationFromId());
        reservationData.setStationToId(offer.getStationToId());
        reservationData.setStationFromCode(fromCode);
        reservationData.setStationToCode(toCode);
        reservationData.setTrainNumber(offer.getTrainInfo().getTrainNumber());
        reservationData.setTrainTicketNumber(offer.getTrainInfo().getTrainTicketNumber());

        reservationData.setTrainTitle(offer.getTrainInfo().getTrainTitle());
        reservationData.setBrandTitle(offer.getTrainInfo().getBrandTitle());
        reservationData.setImInitialStationName(offer.getTrainInfo().getImInitialStationName());
        reservationData.setImFinalStationName(offer.getTrainInfo().getImFinalStationName());
        String[] settlementTitles = getSettlementTitles(offer.getTrainInfo().getTrainTitle());
        reservationData.setStartSettlementTitle(settlementTitles[0]);
        reservationData.setEndSettlementTitle(settlementTitles[1]);

        payload.setCountryFromId(from.getCountryId());
        payload.setCountryToId(to.getCountryId());
        payload.setStationFromTimezone(trainTimeZoneDataProvider.getById(from.getTimeZoneId()).getCode());
        payload.setStationFromRailwayTimezone(trainTimeZoneDataProvider.getById(from.getRailwayTimeZoneId()).getCode());
        payload.setStationToTimezone(trainTimeZoneDataProvider.getById(to.getTimeZoneId()).getCode());
        payload.setStationToRailwayTimezone(trainTimeZoneDataProvider.getById(to.getRailwayTimeZoneId()).getCode());
        payload.setTrainDicts(getTrainDicts(offer));

        var uiData = new TrainReservationUiData();
        payload.setUiData(uiData);
        uiData.setStationFromTitle(from.getTitleDefault());
        uiData.setStationToTitle(to.getTitleDefault());
        uiData.setStationToTitleAccusative(to.getTitleRuAccusativeCase());
        uiData.setStationFromTitleGenitive(from.getTitleRuGenitiveCase());
        uiData.setStationToPreposition(to.getTitleRuPreposition());
        uiData.setStationFromNotGeneralize(from.getNotGeneralize());
        uiData.setStationToNotGeneralize(to.getNotGeneralize());

        if (from.getSettlementId() != 0) {
            var fromSettlement = trainSettlementDataProvider.getById(from.getSettlementId());
            uiData.setStationFromSettlementGeoId(fromSettlement.getGeoId());
            uiData.setStationFromSettlementTitle(fromSettlement.getTitleDefault());
        }
        if (to.getSettlementId() != 0) {
            var toSettlement = trainSettlementDataProvider.getById(to.getSettlementId());
            uiData.setStationToSettlementGeoId(toSettlement.getGeoId());
            uiData.setStationToSettlementTitle(toSettlement.getTitleDefault());
        }

        uiData.setStationFromRailwayTimeZoneText(getTimeZoneTextWithFallback(payload.getStationFromRailwayTimezone()));
        uiData.setStationToRailwayTimeZoneText(getTimeZoneTextWithFallback(payload.getStationToRailwayTimezone()));

        payload.setMoscowRegion(isMoscowRegion(geoId));
        payload.setMobile(isMobile);
        payload.setRoutePolicy(ROUTE_POLICY_MAP.get(offer.getTrainInfo().getRoutePolicy()));
        payload.setCppk(offer.getTrainInfo().getIsCppk());
        payload.setProvider(offer.getTrainInfo().getProvider());
        return payload;
    }

    private TrainDicts getTrainDicts(TTrainServiceOffer offer) {
        var expressNameAliasesMap = new HashMap<Integer, String>();
        List.of(offer.getTrainInfo().getImInitialStationName(), offer.getTrainInfo().getImFinalStationName())
                .forEach(alias -> {
                    if (Strings.isNullOrEmpty(alias)) {
                        return;
                    }
                    Integer stationId = trainStationExpressAliasDataProvider.getStationIdByExpressName(alias);
                    if (stationId == null) {
                        return;
                    }
                    expressNameAliasesMap.put(stationId, alias);
                });

        var stationIds = new HashSet<>(List.of(offer.getStationFromId(), offer.getStationToId()));
        stationIds.addAll(expressNameAliasesMap.keySet());

        var stations = new ArrayList<TrainDicts.Station>();
        for (var id : stationIds) {
            var station = dictStationForId(id);
            station.setExpressName(expressNameAliasesMap.get(id));
            stations.add(station);
        }
        return TrainDicts.builder().stations(stations).build();
    }

    private TrainDicts.Station dictStationForId(int stationId) {
        String stationCode = trainStationCodeDataProvider.getStationExpressCode(stationId).getCode();
        TStation to = trainStationDataProvider.getById(stationId);
        String stationTimezone = trainTimeZoneDataProvider.getById(to.getTimeZoneId()).getCode();
        String stationRailwayTimezone = trainTimeZoneDataProvider.getById(to.getRailwayTimeZoneId()).getCode();

        return TrainDicts.Station.builder()
                .raspID(stationId)
                .expressCode(stationCode)
                .timezone(stationTimezone)
                .railwayTimezone(stationRailwayTimezone)
                .build();
    }

    private String getTimeZoneTextWithFallback(String tzName) {
        try {
            return trainReadableTimezoneDataProvider.getByKey(
                    TrainReadableTimezoneDataProvider.Companion.calculatingByTzKey(DEFAULT_LANGUAGE, tzName)).getValue();
        } catch (NoSuchElementException ex) {
            log.error("ReadableTimezone not found for key: " + tzName, ex);
            EventBuilder eventBuilder = new EventBuilder().withMessage(ex.getMessage())
                    .withLevel(Event.Level.ERROR)
                    .withSentryInterface(new ExceptionInterface(ex))
                    .withTag("vertical", "train");
            Sentry.capture(eventBuilder);
        }
        return tzName;
    }

    private static String[] getSettlementTitles(String trainTitle) {
        String[] rawTitles = trainTitle.split(DASH);
        String[] result = new String[2];
        result[0] = rawTitles[0].strip();
        result[1] = rawTitles[1].strip();
        return result;
    }

    public boolean isMoscowRegion(Integer geoId) {
        var isMoscowRegion = MOSCOW_REGION_GEO_ID.equals(geoId);
        if (!isMoscowRegion) {
            Optional<TSettlement> settlement = trainSettlementDataProvider.getOptionalSettlementByGeoId(geoId);
            if (settlement.isPresent()) {
                isMoscowRegion = MOSCOW_REGION_ID.equals(settlement.get().getRegionId());
            }
        }
        return isMoscowRegion;
    }

    private TrainPassenger convertToTrainPassenger(CreateOrderPassengerV2 requestPassenger, TPassenger offerPassenger) {
        var passenger = new TrainPassenger();
        passenger.setBirthday(requestPassenger.getBirthDate());
        var countryCode = countryDataProvider.getByGeoId(offerPassenger.getCitizenshipGeoId()).getCode();
        passenger.setCitizenshipCode(countryCode);
        passenger.setDocumentNumber(requestPassenger.getDocId());
        passenger.setDocumentType(DOCUMENT_TYPE_MAP.get(offerPassenger.getDocumentType()));
        passenger.setFirstName(requestPassenger.getFirstName());
        passenger.setLastName(requestPassenger.getLastName());
        passenger.setPatronymic(requestPassenger.getPatronymic());
        passenger.setCategory(PASSENGER_CATEGORY_MAP.get(offerPassenger.getPassengerCategory()));
        if (offerPassenger.getLoyaltyCardsList().size() > 0) {
            passenger.setBonusCards(
                    offerPassenger.getLoyaltyCardsList().stream()
                            .map(this::convertToRailwayBonusCard)
                            .collect(Collectors.toList())
            );
        }
        passenger.setSex(SEX_MAP.get(offerPassenger.getSex()));
        passenger.setTariffCode(offerPassenger.getTariffCode());
        passenger.setNonRefundableTariff(offerPassenger.getIsNonRefundableTariff());
        passenger.setRequestedPlaces(offerPassenger.getPlacesList());
        passenger.setPhone(requestPassenger.getPhone());
        passenger.setUsePhoneForReservation(true);
        passenger.setEmail(requestPassenger.getEmail());
        passenger.setUseEmailForReservation(true);
        return passenger;
    }

    private RailwayBonusCard convertToRailwayBonusCard(TLoyaltyCard loyaltyCard) {
        RailwayBonusCard result = new RailwayBonusCard();
        result.setCardNumber(loyaltyCard.getNumber());
        result.setCardType(LOYALTY_CARD_TYPE_MAP.get(loyaltyCard.getType()));
        return result;
    }
}
