package ru.yandex.travel.api.services.orders;

import java.time.Instant;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.travel.api.endpoints.booking_flow.DtoMapper;
import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.api.endpoints.booking_flow.model.FulfillmentType;
import ru.yandex.travel.api.endpoints.booking_flow.model.Guest;
import ru.yandex.travel.api.endpoints.booking_flow.model.HotelOrderDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderStatus;
import ru.yandex.travel.api.models.travel_orders.AviaOrderListItem;
import ru.yandex.travel.api.models.travel_orders.HotelOrderListItem;
import ru.yandex.travel.api.models.travel_orders.HotelOrderListItemPayment;
import ru.yandex.travel.api.models.travel_orders.TrainOrderListItemV2;
import ru.yandex.travel.api.models.travel_orders.TravelOrderListItem;
import ru.yandex.travel.api.services.avia.orders.AviaOrchestratorModelConverter;
import ru.yandex.travel.api.services.hotels_booking_flow.HotelOrdersService;
import ru.yandex.travel.api.services.hotels_booking_flow.models.HotelOrder;
import ru.yandex.travel.api.services.orders.model.TrainOrderListInfo;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.proto.TOrderInfo;

@Service
@RequiredArgsConstructor
public class TravelOrderListItemMapper {
    private final HotelOrdersService hotelOrdersService;
    private final DtoMapper dtoMapper;
    private final AviaOrchestratorModelConverter aviaOrchestratorModelConverter;
    private final TrainModelMapService trainModelMapService;
    private final Map<OrderStatus, FulfillmentType> hotelsFulfillmentMapping = new ImmutableMap.Builder<OrderStatus,
            FulfillmentType>()
            .put(OrderStatus.CONFIRMED, FulfillmentType.COMPLETED)
            .put(OrderStatus.REFUNDED, FulfillmentType.CANCELLED)
            .build();

    public TravelOrderListItem map(TOrderInfo orderInfo) {
        switch (orderInfo.getOrderType()) {
            case OT_HOTEL_EXPEDIA:
                return mapHotelOrder(orderInfo);
            case OT_AVIA_AEROFLOT:
                return mapAviaOrder(orderInfo);
            case OT_TRAIN:
                return mapTrainOrder(orderInfo);
            case OT_GENERIC:
                //noinspection SwitchStatementWithTooFewBranches
                switch (orderInfo.getDisplayOrderType()) {
                    case DT_TRAIN:
                        return mapTrainOrder(orderInfo);
                    default:
                        throw new RuntimeException("Unsupported generic order display type: "
                                + orderInfo.getDisplayOrderType());
                }
            default:
                throw new RuntimeException("Unknown order type " + orderInfo.getOrderType());
        }
    }

    private HotelOrderListItem mapHotelOrder(TOrderInfo orderInfo) {
        HotelOrder orderModel = hotelOrdersService.getOrderFromProto(orderInfo);
        HotelOrderDto order = dtoMapper.buildOrderDto(orderModel, false);
        HotelOrderListItem item = new HotelOrderListItem();
        item.setId(order.getId().toString());
        item.setYandexOrderId(order.getYandexOrderId());
        item.setHotelName(order.getOrderInfo().getPartnerHotelInfo().getName());
        if (order.getOrderInfo().getPartnerHotelInfo().getRatings() != null &&
                order.getOrderInfo().getPartnerHotelInfo().getRatings().getProperty().getType().equals("Star")) {
            item.setNumStars((int) Double.parseDouble(order.getOrderInfo().getPartnerHotelInfo().getRatings().getProperty().getRating()));
        } else {
            item.setNumStars(order.getOrderInfo().getBasicHotelInfo().getStars());
        }
        item.setCheckinDate(order.getOrderInfo().getRequestInfo().getCheckinDate());
        item.setCheckoutDate(order.getOrderInfo().getRequestInfo().getCheckoutDate());
        item.setGuests(order.getGuestInfo().getGuests().stream()
                .map(g -> new Guest(g.getFirstName(), g.getLastName()))
                .collect(Collectors.toList()));
        item.setStatus(DisplayableOrderStatus.fromProto(orderInfo.getDisplayOrderState()));
        item.setOrderType(OrderType.HOTEL);
        item.setFulfillmentType(hotelsFulfillmentMapping.get(order.getStatus()));
        if (order.getOrderPriceInfo() != null) {
            item.setPrice(DtoMapper.moneyToRateDto(order.getOrderPriceInfo().getPrice()));
        } else {
            item.setPrice(order.getOrderInfo().getRateInfo().getHotelCharges().getTotals().getGrand());
        }
        item.setServicedAt(ProtoUtils.toLocalDateTime(orderInfo.getServicedAt()));
        item.setSource(OrderListSource.ORCHESTRATOR);
        if (orderModel.getPayment() != null) {
            HotelOrderListItemPayment.Next next = null;
            if (orderModel.getPayment().getNext() != null &&
                    orderModel.getPayment().getNext().getPaymentEndsAt() != null && orderModel.getPayment().getNext().getPaymentEndsAt().isAfter(Instant.now())) {
                next = HotelOrderListItemPayment.Next.builder()
                        .amount(orderModel.getPayment().getNext().getAmount())
                        .paymentEndsAt(orderModel.getPayment().getNext().getPaymentEndsAt())
                        .penaltyIfUnpaid(orderModel.getPayment().getNext().getPenaltyIfUnpaid())
                        .build();
            }

            item.setPayment(HotelOrderListItemPayment.builder()
                    .amountPaid(orderModel.getPayment().getAmountPaid())
                    .usesDeferredPayments(orderModel.getPayment().isUsesDeferredPayment())
                    .next(next)
                    .mayBeStarted(orderModel.getPayment().isMayBeStarted())
                    .build());
        }
        return item;
    }

    @SuppressWarnings("Duplicates")
    private AviaOrderListItem mapAviaOrder(TOrderInfo orderInfo) {
        ensureAviaExpectedOrderStructure(orderInfo);
        OrderDTO convertedOrder = aviaOrchestratorModelConverter.fromProto(orderInfo, AeroflotServicePayload.class);
        AviaOrderListItem item = new AviaOrderListItem();
        item.setId(convertedOrder.getId());
        item.setYandexOrderId(convertedOrder.getPrettyId());
        item.setOrderType(OrderType.AVIA);
        item.setStatus(DisplayableOrderStatus.fromProto(convertedOrder.getEDisplayOrderState()));
        item.setAirReservation(convertedOrder.getAirReservation());
        item.setTravellers(convertedOrder.getTravellers());
        item.setPrice(convertedOrder.getPrice());
        item.setState(convertedOrder.getState());
        item.setTimeLimitAt(convertedOrder.getTimeLimitAt());
        item.setErrorCode(convertedOrder.getErrorCode());
        item.setReference(convertedOrder.getReference());
        item.setServicedAt(convertedOrder.getServicedAt());
        item.setSource(OrderListSource.ORCHESTRATOR);
        return item;
    }


    private TrainOrderListItemV2 mapTrainOrder(TOrderInfo orderInfo) {
        TrainOrderListItemV2 item = new TrainOrderListItemV2();
        TrainOrderListInfo trainOrderListInfo = trainModelMapService.convertToListInfo(orderInfo);
        item.setOrderInfo(trainOrderListInfo);
        item.setId(orderInfo.getOrderId());
        item.setYandexOrderId(orderInfo.getPrettyId());

        // TODO (mbobrov): determine train order status
        item.setStatus(DisplayableOrderStatus.fromProto(orderInfo.getDisplayOrderState()));
        item.setServicedAt(ProtoUtils.toLocalDateTime(orderInfo.getServicedAt()));
        item.setOrderType(OrderType.TRAIN);
        item.setSource(OrderListSource.ORCHESTRATOR);
        return item;
    }

    private void ensureAviaExpectedOrderStructure(TOrderInfo order) {
        Preconditions.checkArgument(order.getServiceCount() == 1,
                "exactly 1 service is expected instead of %s", order.getServiceCount());
        Preconditions.checkArgument(order.getInvoiceCount() <= 1,
                "no more than 1 invoice is expected instead of %s", order.getInvoiceCount());
    }

}
