package ru.yandex.travel.api.services.orders.authorization;

import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import io.grpc.Context;
import lombok.RequiredArgsConstructor;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.endpoints.travel_orders.req_rsp.OrderAuthorizationRspV1;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.proto.TAuthorizeForOrderReq;

@RequiredArgsConstructor
public class AuthorizationService {
    private final AuthorizationClientFactory authorizationClientFactory;
    private final List<OrderDigestProvider> digestProviders;
    private final Retry retryHelper;

    private static String ensureIdIsUid(String id) {
        try {
            //noinspection ResultOfMethodCallIgnored
            UUID.fromString(id);
            return id;
        } catch (IllegalArgumentException ignored) {
            return id.replaceFirst("(\\p{XDigit}{8})(\\p{XDigit}{4})(\\p{XDigit}{4})(\\p{XDigit}{4})(\\p{XDigit}+)",
                    "$1-$2-$3-$4-$5");
        }
    }

    @SuppressWarnings("Duplicates")
    public CompletableFuture<OrderAuthorizationRspV1> checkAuthorization(String orderId, String prettyId) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(orderId) ^ StringUtils.isNotBlank(prettyId),
                "Exactly one of orderId or prettyId must be specified");
        var digestFutures = StringUtils.isNotEmpty(orderId) ? getDigestByOrderId(orderId) :
                getDigestByPrettyId(prettyId);
        var credentials = UserCredentials.get();
        Preconditions.checkArgument(credentials != null, "User credentials must be passed to use this service");
        return retryHelper.withRetry("AuthorizationService::checkAuthorization",
                () -> getOneDigest(digestFutures)
                        .thenCompose(digest -> checkAuthorizationImpl(digest, credentials)),
                //TODO (mbobrov): process http errors from train-api here
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @SuppressWarnings("Duplicates")
    public CompletableFuture<OrderAuthorizationRspV1> requestAuthorization(String orderId, String prettyId,
                                                                           String secret) {
        Preconditions.checkArgument(StringUtils.isNotEmpty(orderId) ^ StringUtils.isNotBlank(prettyId),
                "Exactly one of orderId or prettyId must be specified");
        var digestFutures = StringUtils.isNotEmpty(orderId) ? getDigestByOrderId(orderId) :
                getDigestByPrettyId(prettyId);
        var credentials = UserCredentials.get();
        Preconditions.checkArgument(credentials != null, "User credentials must be passed to use this service");
        return retryHelper.withRetry("AuthorizationService::requestAuthorization",
                () -> getOneDigest(digestFutures)
                        .thenCompose(digest -> requestAuthorizationImpl(digest, secret, credentials)),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    private List<CompletableFuture<OrderDigest>> getDigestByPrettyId(String prettyId) {
        return digestProviders.stream().map(provider -> provider.getByPrettyId(prettyId)).collect(Collectors.toList());
    }

    private List<CompletableFuture<OrderDigest>> getDigestByOrderId(String orderId) {
        return digestProviders.stream().map(provider -> provider.getByOrderId(orderId)).collect(Collectors.toList());
    }

    private CompletableFuture<OrderAuthorizationRspV1> checkAuthorizationImpl(OrderDigest digest,
                                                                              UserCredentials credentials) {
        var inplaceAuth = authorizeInplace(digest, credentials);
        if (inplaceAuth != null) {
            return inplaceAuth;
        }
        // We need to restore initial credentials since we could loose them as we are in other thread now
        Context.current().withValue(UserCredentials.KEY, credentials).attach();
        return FutureUtils.buildCompletableFuture(authorizationClientFactory.createFutureStub()
                .checkAuthorization(TAuthorizeForOrderReq.newBuilder()
                        .setOrderId(ensureIdIsUid(digest.getOrderId()))
                        .build()))
                .thenApply(resp -> new OrderAuthorizationRspV1(digest.getOrderId(), digest.getPrettyId(),
                        resp.getAuthorized(), digest.getType()));
    }

    private CompletableFuture<OrderAuthorizationRspV1> requestAuthorizationImpl(OrderDigest digest, String secret,
                                                                                UserCredentials credentials) {
        var inplaceAuth = authorizeInplace(digest, credentials);
        if (inplaceAuth != null) {
            return inplaceAuth;
        }
        if (digest.matchesSecret(secret)) {
            // We need to restore initial credentials since we could loose them as we are in other thread now
            Context.current().withValue(UserCredentials.KEY, credentials).attach();
            return FutureUtils.buildCompletableFuture(authorizationClientFactory.createFutureStub()
                    .authorize(TAuthorizeForOrderReq.newBuilder()
                            .setOrderId(ensureIdIsUid(digest.getOrderId()))
                            .build()))
                    .thenApply(resp -> new OrderAuthorizationRspV1(digest.getOrderId(),
                            digest.getPrettyId(), true, digest.getType()));
        } else {
            return CompletableFuture.completedFuture(
                    new OrderAuthorizationRspV1(digest.getOrderId(), digest.getPrettyId(), false,
                            digest.getType())
            );
        }
    }


    private CompletableFuture<OrderAuthorizationRspV1> authorizeInplace(OrderDigest digest,
                                                                        UserCredentials credentials) {
        if (digest == null) {
            return CompletableFuture.completedFuture(null);
        }
        if (digest.matchesOwner(credentials)) {
            return CompletableFuture.completedFuture(new OrderAuthorizationRspV1(digest.getOrderId(),
                    digest.getPrettyId(), true, digest.getType()));
        }
        return null;
    }

    private CompletableFuture<OrderDigest> getOneDigest(List<CompletableFuture<OrderDigest>> digestFutures) {
        return CompletableFuture.allOf(digestFutures.toArray(new CompletableFuture[0]))
                .thenApply(v -> digestFutures.stream()
                        .map(CompletableFuture::join)
                        .filter(Objects::nonNull)
                        .findFirst().orElse(null));
    }
}
