package ru.yandex.travel.api.services.orders.authorization;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.services.orders.OrchestratorClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.proto.TGetUnauthorizedOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetUnauthorizedOrderInfoRsp;

@Service
@RequiredArgsConstructor
@Slf4j
public class OrchestratorOrderDigestProvider implements OrderDigestProvider {
    private final OrchestratorClientFactory clientFactory;

    @Override
    public CompletableFuture<OrderDigest> getByPrettyId(String prettyId) {
        var listenableFuture = clientFactory.createOrderNoAuthFutureStub().getUnauthorizedOrderInfo(
                TGetUnauthorizedOrderInfoReq.newBuilder().setPrettyId(prettyId).build());
        return FutureUtils.buildCompletableFuture(listenableFuture)
                .thenApply(this::buildDigest)
                .exceptionally(ex -> {
                    log.warn("Order digest retrieval has failed: order id {}", prettyId, ex);
                    return null;
                });
    }

    @Override
    public CompletableFuture<OrderDigest> getByOrderId(String orderId) {
        if (!isValidUuid(orderId)) {
            return CompletableFuture.completedFuture(null);
        }
        var listenableFuture = clientFactory.createOrderNoAuthFutureStub().getUnauthorizedOrderInfo(
                TGetUnauthorizedOrderInfoReq.newBuilder().setOrderId(orderId).build());
        return FutureUtils.buildCompletableFuture(listenableFuture).thenApply(this::buildDigest).exceptionally(ex -> null);
    }

    private boolean isValidUuid(String value) {
        try {
            //noinspection ResultOfMethodCallIgnored
            UUID.fromString(value);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    private OrderDigest buildDigest(TGetUnauthorizedOrderInfoRsp resp) {
        var digest = new OrderDigest();
        digest.setEmailHash(resp.getInfo().getEmailHash());
        digest.setPhoneHash(resp.getInfo().getPhoneHash());
        digest.setPrettyId(resp.getInfo().getPrettyId());
        digest.setOrderId(resp.getInfo().getOrderId());
        digest.setType(fromEOrderType(resp.getInfo().getOrderType()));
        if (StringUtils.isNotBlank(resp.getInfo().getOwnerPassportId())) {
            digest.setOwnerPassportId(resp.getInfo().getOwnerPassportId());
        }
        return digest;
    }

    private AuthorizationOrderType fromEOrderType(EOrderType eOrderType) {
        switch (eOrderType) {
            case OT_AVIA_AEROFLOT:
                return AuthorizationOrderType.OT_AVIA_AEROFLOT;
            case OT_HOTEL_EXPEDIA:
                return AuthorizationOrderType.OT_HOTEL_EXPEDIA;
            case OT_TRAIN:
                return AuthorizationOrderType.OT_TRAIN_ORCH;
            case OT_BUS:
                return AuthorizationOrderType.OT_BUS;
            case OT_GENERIC:
                return AuthorizationOrderType.OT_GENERIC;
            default:
                throw new IllegalArgumentException(String.format("Don't know how to map %s", eOrderType));
        }
    }
}
