package ru.yandex.travel.api.services.orders.happy_page;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.MoreExecutors;
import lombok.Builder;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.travel_orders.req_rsp.OrderHappyPageRspV1;
import ru.yandex.travel.api.services.orders.OrderType;
import ru.yandex.travel.api.services.orders.happy_page.model.CrossSaleBlock;
import ru.yandex.travel.api.services.orders.happy_page.model.CrossSaleInfo;
import ru.yandex.travel.api.services.orders.happy_page.model.HappyPageOrder;

@Slf4j
@Service
public class HappyPageBlockCollector {
    private final ScheduledExecutorService executor;

    public HappyPageBlockCollector() {
        this.executor = Executors.newSingleThreadScheduledExecutor();
    }

    @PreDestroy
    public void shutdown() {
        MoreExecutors.shutdownAndAwaitTermination(this.executor, 100, TimeUnit.MILLISECONDS);
    }

    @Value
    @Builder
    public static class CollectorData {
        AtomicInteger counter;
        OrderHappyPageRspV1.OrderHappyPageRspV1Builder responseCollector;
        CrossSaleInfo.CrossSaleInfoBuilder crossSaleBlocksCollector;
        CompletableFuture<OrderHappyPageRspV1> responseFuture;

        public void tickCounter() {
            if (counter.decrementAndGet() == 0) {
                completeResponseFuture();
            }
        }

        public void completeResponseFuture() {
            responseFuture.complete(
                    responseCollector
                            .crossSale(crossSaleBlocksCollector.build())
                            .build()
            );
        }
    }

    public CompletableFuture<OrderHappyPageRspV1> getHappyPageResponseFuture(HappyPageOrder orderInfo,
                                                                             OrderType orderType,
                                                                             List<CompletableFuture<CrossSaleBlock>> crossSaleBlockFutures,
                                                                             Duration timeoutDuration) {
        var data = CollectorData.builder()
                .counter(new AtomicInteger(crossSaleBlockFutures.size()))
                .responseFuture(new CompletableFuture<>())
                .responseCollector(OrderHappyPageRspV1.builder()
                        .order(orderInfo)
                        .orderType(orderType))
                .crossSaleBlocksCollector(CrossSaleInfo.builder())
                .build();

        for (CompletableFuture<CrossSaleBlock> blockFuture : crossSaleBlockFutures) {
            blockFuture.handle((r, t) -> {
                if (t != null) {
                    log.error("Couldn't fetch cross sale block info", t);
                }
                return r;
            }).whenComplete((r, t) -> {
                 if (r != null) {
                    data.getCrossSaleBlocksCollector().block(r);
                    data.tickCounter();
                }
            });
        }

        executor.schedule(data::completeResponseFuture, timeoutDuration.toMillis(), TimeUnit.MILLISECONDS);

        return data.getResponseFuture();
    }
}
