package ru.yandex.travel.api.services.orders.happy_page;

import java.time.Instant;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.travel.api.endpoints.booking_flow.model.Guest;
import ru.yandex.travel.api.endpoints.booking_flow.model.HotelOrderDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OfferDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderGuestInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentDto;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.OrderInfoRspV1;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.services.orders.happy_page.model.AviaHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.BusHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.BusHappyPagePaymentInfo;
import ru.yandex.travel.api.services.orders.happy_page.model.HotelHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.HotelOrderInfo;
import ru.yandex.travel.api.services.orders.happy_page.model.OrderPaymentInfo;
import ru.yandex.travel.api.services.orders.happy_page.model.OrderPaymentInfoPendingPayment;
import ru.yandex.travel.api.services.orders.happy_page.model.OrderPaymentInfoReceiptItem;
import ru.yandex.travel.api.services.orders.happy_page.model.ReceiptItemType;
import ru.yandex.travel.api.services.orders.happy_page.model.TrainHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.TrainOrderError;

@Service
@RequiredArgsConstructor
public class HappyPageMapper {
    public AviaHappyPageOrder mapOrderDto(OrderDTO orderDTO) {
        var aviaBuilder = AviaHappyPageOrder.builder();
        aviaBuilder.id(UUID.fromString(orderDTO.getId()))
                .prettyId(orderDTO.getPrettyId())
                .email(orderDTO.getEmail())
                .phone(orderDTO.getPhone())
                .partner(orderDTO.getPartner())
                .variantId(orderDTO.getVariantId())
                .airReservation(orderDTO.getAirReservation())
                .travellers(orderDTO.getTravellers())
                .price(priceFromMoney(orderDTO.getPrice()))
                .preliminaryPrice(priceFromMoney(orderDTO.getPreliminaryPrice()))
                .state(orderDTO.getState())
                .timeLimitAt(orderDTO.getTimeLimitAt())
                .errorCode(orderDTO.getErrorCode())
                .reference(orderDTO.getReference())
                .eDisplayOrderState(orderDTO.getEDisplayOrderState())
                .servicedAt(orderDTO.getServicedAt())
                .promoCampaigns(orderDTO.getPromoCampaigns());
        return aviaBuilder.build();
    }

    private Price priceFromMoney(Money money) {
        if (money == null) {
            return null;
        }
        Price result = new Price();
        result.setCurrency(Price.Currency.RUB);
        result.setValue(money.getNumberStripped().intValue());
        return result;
    }

    public TrainHappyPageOrder mapTrainModel(OrderInfoRspV1 convert) {
        var rspBuilder = TrainHappyPageOrder.builder()
                .arrival(convert.getArrival())
                .canChangeElectronicRegistrationTill(Instant.now())
                .carNumber(convert.getCarNumber())
                .carType(convert.getCarType())
                .coachOwner(convert.getCoachOwner())
                .companyTitle(convert.getCompanyTitle())
                .compartmentGender(convert.getCompartmentGender())
                .customerEmail(convert.getCustomerEmail())
                .customerPhone(convert.getCustomerPhone())
                .departure(convert.getDeparture())
                .id(convert.getId())
                .insuranceStatus(convert.getInsuranceStatus())
                .maxPendingTill(convert.getMaxPendingTill())
                .onlyFullReturnPossible(false)
                .partner(convert.getPartner())
                .partnerOrderId(convert.getPartnerOrderId())
                .paymentError(convert.getPaymentError())
                .paymentReceiptUrl(convert.getPaymentReceiptUrl())
                .paymentUrl(convert.getPaymentUrl())
                .prettyId(convert.getPrettyId())
                // todo(tlg-13,ganintsev): removed these fields and onlyFullReturnPossible?
                .rebookingAvailable(false)
                .reservationProlonged(false)
                .reservedTo(convert.getReservedTo())
                .specialNotice(convert.getSpecialNotice())
                .stationFrom(convert.getStationFrom())
                .stationTo(convert.getStationTo())
                .status(convert.getStatus())
                .trainInfo(convert.getTrainInfo())
                .twoStorey(convert.isTwoStorey())
                .segments(convert.getSegments());
        if (convert.getPassengers() != null) {
            rspBuilder.passengers(convert.getPassengers());
        }
        if (convert.getWarnings() != null) {
            rspBuilder.warnings(convert.getWarnings());
        }
        if (convert.getError() != null) {
            var error = convert.getError();
            TrainOrderError.TrainOrderErrorBuilder errorBuilder = TrainOrderError.builder()
                    .type(error.getType())
                    .message(error.getMessage())
                    .messageCode(error.getMessageCode());
            if (error.getMessageParams() != null && error.getMessageParams().size() > 0) {
                errorBuilder.messageParams(error.getMessageParams());
            }
            rspBuilder.error(errorBuilder.build());
        }
        return rspBuilder.build();
    }

    public HotelHappyPageOrder mapHotelOrder(HotelOrderDto buildOrderDto) {
        OrderGuestInfo orderGuestInfo = new OrderGuestInfo();
        orderGuestInfo.setCustomerEmail(buildOrderDto.getGuestInfo().getCustomerEmail());
        orderGuestInfo.setCustomerPhone(buildOrderDto.getGuestInfo().getCustomerPhone());
        orderGuestInfo.setAllowsSubscription(buildOrderDto.getGuestInfo().isAllowsSubscription());

        orderGuestInfo.setGuests(buildOrderDto.getGuestInfo().getGuests().stream()
                .map(g -> new Guest(g.getFirstName(), g.getLastName()))
                .collect(Collectors.toList()));

        HotelHappyPageOrder.HotelHappyPageOrderBuilder rspBuilder = HotelHappyPageOrder.builder()
                .id(buildOrderDto.getId())
                .confirmationInfo(buildOrderDto.getConfirmationInfo())
                .guestInfo(orderGuestInfo)
                .orderCancellationDetails(buildOrderDto.getOrderCancellationDetails())
                .refundInfo(buildOrderDto.getRefundInfo())
                .status(buildOrderDto.getStatus())
                .yandexOrderId(buildOrderDto.getYandexOrderId());
        if (buildOrderDto.getOrderInfo() != null) {
            OfferDto offerDto = buildOrderDto.getOrderInfo();
            rspBuilder.orderInfo(HotelOrderInfo.builder()
                    .basicHotelInfo(offerDto.getBasicHotelInfo())
                    .bedGroups(offerDto.getBedGroups())
                    .cancellationInfo(offerDto.getCancellationInfo())
                    .cancellationInfoUnfiltered(offerDto.getCancellationInfoUnfiltered())
                    .checksum(offerDto.getChecksum())
                    .directPartner(offerDto.getDirectPartner())
                    .label(offerDto.getLabel())
                    .legalInfo(offerDto.getLegalInfo())
                    .pansionInfo(offerDto.getPansionInfo())
                    .partnerHotelInfo(offerDto.getPartnerHotelInfo())
                    .partnerId(offerDto.getPartnerId())
                    .partnerRoomInfo(offerDto.getPartnerRoomInfo())
                    .promoCampaigns(offerDto.getPromoCampaigns())
                    .rateInfo(offerDto.getRateInfo())
                    .requestInfo(offerDto.getRequestInfo())
                    .sessionKey(offerDto.getSessionKey())
                    .travelToken(offerDto.getTravelToken())
                    .build());
        }
        if (buildOrderDto.getPayment() != null) {
            PaymentDto payment = buildOrderDto.getPayment();
            OrderPaymentInfo.OrderPaymentInfoBuilder paymentInfoBuilder = OrderPaymentInfo.builder()
                    .usesDeferredPayments(payment.isUsesDeferredPayments())
                    .amountPaid(payment.getAmountPaid());
            if (payment.getError() != null) {
                paymentInfoBuilder.errorInfo(payment.getError().getCode());
            }
            if (payment.getNext() != null && payment.getNext().getPaymentEndsAt() != null && payment.getNext().getPaymentEndsAt().isAfter(Instant.now())) {
                paymentInfoBuilder.next(OrderPaymentInfoPendingPayment.builder()
                        .amount(payment.getNext().getAmount())
                        .paymentEndsAt(payment.getNext().getPaymentEndsAt())
                        .penaltyIfUnpaid(payment.getNext().getPenaltyIfUnpaid())
                        .build());
            }
            if (payment.getReceipts() != null) {
                paymentInfoBuilder.receipts(
                        payment.getReceipts().stream()
                                .map(receipt -> OrderPaymentInfoReceiptItem.builder()
                                        .type(ReceiptItemType.fromProto(receipt.getType()))
                                        .url(receipt.getUrl())
                                        .build())
                                .collect(Collectors.toList())
                );
            }
            rspBuilder.payment(paymentInfoBuilder.build());
        }
        rspBuilder.orderPriceInfo(buildOrderDto.getOrderPriceInfo());
        rspBuilder.appliedPromoCampaigns(buildOrderDto.getAppliedPromoCampaigns());
        return rspBuilder.build();
    }

    public BusHappyPageOrder mapBusOrder(GetGenericOrderRspV1 order) {
        var builder = BusHappyPageOrder.builder();
        builder
                .id(order.getId())
                .prettyId(order.getPrettyId())
                .state(order.getState())
                .cancellationReason(order.getCancellationReason())
                .displayState(order.getDisplayState())
                .expiresAt(order.getExpiresAt())
                .servicedAt(order.getServicedAt())
                .services(order.getServices())
                .orderPriceInfo(order.getOrderPriceInfo())
                .contactInfo(order.getContactInfo())
                .refundPartInfo(order.getRefundPartInfo())
        ;
        var paymentBuilder = BusHappyPagePaymentInfo.builder();
        paymentBuilder
                .paymentUrl(order.getPayment().getPaymentUrl())
                .errorInfo(order.getPayment().getErrorInfo())
                .receipts(order.getPayment().getReceipts())
        ;
        builder.payment(paymentBuilder.build());
        return builder.build();
    }

}
