package ru.yandex.travel.api.services.orders.happy_page;

import java.math.BigDecimal;
import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.travel_orders.req_rsp.OrderHappyPageRspV1;
import ru.yandex.travel.api.services.avia.api_gateway.AviaApiGatewayService;
import ru.yandex.travel.api.services.buses.BusesFacadeService;
import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.orders.OrchestratorClientFactory;
import ru.yandex.travel.api.services.orders.OrderType;
import ru.yandex.travel.api.services.orders.happy_page.afisha.AfishaService;
import ru.yandex.travel.api.services.orders.happy_page.afisha.AfishaServiceProperties;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaCitiesResponse;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaImageSize;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.IziTravelProperties;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.IziTravelService;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelCitiesChildrenResponse;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelHelper;
import ru.yandex.travel.api.services.orders.happy_page.model.AfishaCrossSalePayload;
import ru.yandex.travel.api.services.orders.happy_page.model.AviaHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.CrossSaleBlock;
import ru.yandex.travel.api.services.orders.happy_page.model.CrossSaleBlockType;
import ru.yandex.travel.api.services.orders.happy_page.model.HappyPageLogRecord;
import ru.yandex.travel.api.services.orders.happy_page.model.HappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.HotelHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.IziTravelCrossSalePayload;
import ru.yandex.travel.api.services.orders.happy_page.model.PromoCodePayload;
import ru.yandex.travel.api.services.orders.happy_page.model.PromoPayload;
import ru.yandex.travel.api.services.orders.happy_page.model.TrainHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.TransportPayload;
import ru.yandex.travel.api.services.orders.happy_page.morda_backend.MordaBackendService;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.proto.TGeneratedPromoCodesInfo;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;

@Slf4j
@Service
@RequiredArgsConstructor
@EnableConfigurationProperties({HappyPageProperties.class, AfishaServiceProperties.class, IziTravelProperties.class})
public class HappyPageService {
    private static final Logger HP_LOGGER = LoggerFactory.getLogger("ru.yandex.travel.happy_page");
    private final TrainSettlementDataProvider trainSettlementDataProvider;
    private final HappyPageProperties config;
    private final HappyPageBlockCollector blockCollector;
    private final OrchestratorClientFactory orchestratorClientFactory;
    private final MordaBackendService mordaBackendService;
    private final BusesFacadeService busesFacadeService;
    private final RegionsService regionsService;
    private final AviaApiGatewayService apiGatewayService;
    private final AfishaService afishaService;
    private final AfishaServiceProperties afishaProperties;
    private final AfishaResponseMapper afishaResponseMapper;
    private final IziTravelService iziTravelService;
    private final IziTravelProperties iziTravelProperties;
    private final IziTravelResponseMapper iziTravelResponseMapper;
    private final ObjectMapper objectMapper;
    private final Clock clock;
    private final HotelPayloadProvider hotelPayloadProvider;
    private final MarketWidgetPayloadProvider marketWidgetPayloadProvider;
    private final OrderGeoIDExtractor orderGeoIDExtractor;
    private final OrderFromProtoMapper orderFromProtoMapper;

    public CompletableFuture<OrderHappyPageRspV1> getOrderHappyPage(UUID orderId,
                                                                    String geoId,
                                                                    CommonHttpHeaders headers,
                                                                    UserCredentials userCredentials) {
        var requestTime = Instant.now(clock);
        var logRecord = new HappyPageLogRecord();
        logRecord.setOrderId(orderId.toString());
        logRecord.setLogId(orderId.toString());
        logRecord.setTimestamp(requestTime.toEpochMilli());
        // prepare main info
        TGetOrderInfoReq req = TGetOrderInfoReq.newBuilder()
                .setOrderId(orderId.toString())
                .build();
        AtomicReference<OrderType> orderType = new AtomicReference<>();
        TGetOrderInfoRsp protoOrderInfo = FutureUtils.buildCompletableFuture(
                orchestratorClientFactory.createFutureStubForHappyPage().getOrderInfo(req))
                .whenComplete((r, t) -> {
                    if (r != null) {
                        logRecord.setOrderPrettyId(r.getResult().getPrettyId());
                        orderType.set(OrderType.fromProto(r.getResult().getDisplayOrderType()));
                        r.getResult().getServiceList().forEach(service ->
                                logRecord.addServiceType(service.getServiceType().toString()));
                    }
                })
                .join();
        HappyPageOrder orderInfo = orderFromProtoMapper.map(protoOrderInfo, false);
        // prepare cross sale blocks
        var defaultBlock = new HappyPageProperties.BlockSettings();
        defaultBlock.setOrder(1);
        defaultBlock.setType(CrossSaleBlockType.PROMO);
        defaultBlock.setPromoAdFoxId("");
        List<HappyPageProperties.BlockSettings> pageSchemas = config.getPageSchemas().getOrDefault(orderType.get(),
                List.of(defaultBlock));

        // TRAVELBACK-2078: do not request cross sale block for cancelled avia orders
        if (isNotConfirmedAviaOrder(orderInfo)) {
            pageSchemas = List.of(defaultBlock);
        }

        List<CompletableFuture<CrossSaleBlock>> crossSaleBlockFutures = new ArrayList<>();
        for (HappyPageProperties.BlockSettings blockSettings : pageSchemas) {
            CompletableFuture<CrossSaleBlock> blockFuture = getCrossSaleBlockInfoSafe(orderInfo, orderType.get(),
                    blockSettings, headers, userCredentials, geoId, protoOrderInfo)
                    .whenComplete((response, error) -> {
                        HappyPageLogRecord.BlockInfo logBlock = new HappyPageLogRecord.BlockInfo();
                        logBlock.setType(blockSettings.getType().toString());
                        logBlock.setShown(true);
                        logBlock.setResponseTime(Instant.now(clock).toEpochMilli() - requestTime.toEpochMilli());
                        if (error != null) {
                            logBlock.setShown(false);
                            logBlock.setErrorInfo(error.getMessage());
                        }
                        logRecord.addBlock(logBlock);
                    });
            crossSaleBlockFutures.add(blockFuture);
        }
        // collect all info together
        return blockCollector.getHappyPageResponseFuture(orderInfo, orderType.get(), crossSaleBlockFutures,
                config.getLoadBlockTimeout())
                .whenComplete((response, error) -> {
                    Instant responseTime = Instant.now(clock);
                    if (response != null) {
                        logRecord.setOrderType(response.getOrderType().toString());
                        logRecord.setResponseBody(response);
                    }
                    if (error != null) {
                        logRecord.setErrorInfo(error.getMessage());
                    }
                    logRecord.setResponseTime(responseTime.toEpochMilli() - requestTime.toEpochMilli());
                    try {
                        HP_LOGGER.info(objectMapper.writerFor(HappyPageLogRecord.class).writeValueAsString(logRecord));
                    } catch (JsonProcessingException e) {
                        log.error("Failed to record happy-page log for order" + logRecord.getOrderId(), e);
                    }
                });
    }

    private CompletableFuture<CrossSaleBlock> getCrossSaleBlockInfoSafe(HappyPageOrder order,
                                                                        OrderType orderType,
                                                                        HappyPageProperties.BlockSettings blockSettings,
                                                                        CommonHttpHeaders headers,
                                                                        UserCredentials userCredentials,
                                                                        String fromGeoId,
                                                                        TGetOrderInfoRsp protoOrderInfo) {
        try {
            return getCrossSaleBlockInfo(order, orderType, blockSettings, headers, userCredentials, fromGeoId, protoOrderInfo);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }
    private CompletableFuture<CrossSaleBlock> getCrossSaleBlockInfo(HappyPageOrder order,
                                                                    OrderType orderType,
                                                                    HappyPageProperties.BlockSettings blockSettings,
                                                                    CommonHttpHeaders headers,
                                                                    UserCredentials userCredentials,
                                                                    String fromGeoId,
                                                                    TGetOrderInfoRsp protoOrderInfo) {
        CrossSaleBlock.CrossSaleBlockBuilder blockInfoBuilder =
                CrossSaleBlock.builder().blockType(blockSettings.getType());
        switch (blockSettings.getType()) {
            case HOTEL_CROSS_SALE:
                var geoId = orderGeoIDExtractor.extract(order);
                return hotelPayloadProvider.get(order, geoId, blockSettings, headers, userCredentials, "HappyPage")
                        .orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS)
                        .thenApply(hotelPayload -> blockInfoBuilder
                                .uiPayload(hotelPayload)
                                .order(blockSettings.getOrder())
                                .build());
            case PROMO:
                PromoPayload promoPayload = new PromoPayload(blockSettings.getPromoAdFoxId());
                return CompletableFuture.supplyAsync(() -> blockInfoBuilder
                        .uiPayload(promoPayload)
                        .order(blockSettings.getOrder())
                        .build())
                        .orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
            case TRANSPORT_CROSS_SALE:
                TransportPayload transportPayload = new TransportPayload();
                int toGeoId = orderGeoIDExtractor.extract(order);
                var toRegion = regionsService.getRegion(toGeoId, "ru", "ru");
                if (Strings.isNullOrEmpty(fromGeoId)) {
                    transportPayload.setRegionToLinguistics(toRegion.getLinguistics());
                    var emptyBlocks = List.of(
                            TransportPayload.TransportBlockInfo.emptyBlock(TransportPayload.TransportBlockType.AVIA),
                            TransportPayload.TransportBlockInfo.emptyBlock(TransportPayload.TransportBlockType.TRAIN),
                            TransportPayload.TransportBlockInfo.emptyBlock(TransportPayload.TransportBlockType.BUS)
                    );
                    transportPayload.setTransportBlocks(emptyBlocks);
                    return CompletableFuture.supplyAsync(
                            () -> blockInfoBuilder
                                    .uiPayload(transportPayload)
                                    .order(blockSettings.getOrder())
                                    .build()
                    ).orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
                } else {
                    int parsedFromGeoId = Integer.parseInt(fromGeoId);
                    if (parsedFromGeoId == toGeoId) {
                        return CompletableFuture.failedFuture(
                                new IllegalArgumentException("GeoIds of src and dst are the same: " + fromGeoId)
                        );
                    }
                    var fromRegion = regionsService.getRegion(parsedFromGeoId, "ru", "ru");
                    transportPayload.setRegionFromLinguistics(fromRegion.getLinguistics());
                    transportPayload.setRegionToLinguistics(toRegion.getLinguistics());
                    List<TransportPayload.TransportBlockInfo> blocks = new ArrayList<>();
                    TransportPayload.TransportBlockInfo blockInfo = getTransportRequestParamsFromOrder(order);
                    CompletableFuture<Boolean> trainTransferExists = mordaBackendService.checkTrainTransferExists(
                            fromRegion.getGeoId(),
                            toRegion.getGeoId(),
                            blockInfo.getDepartureAt()
                    );
                    CompletableFuture<Boolean> busTransferExists = busesFacadeService.checkBusTransferExists(
                            fromRegion.getGeoId(),
                            toRegion.getGeoId(),
                            blockInfo.getDepartureAt()
                    );
                    CompletableFuture<Boolean> aviaTransferExists = apiGatewayService.isPossibleTrip(
                            fromRegion.getGeoId(),
                            toRegion.getGeoId(),
                            blockInfo.getDepartureAt()
                    );
                    return CompletableFuture.supplyAsync(() -> {
                        try {
                            if (aviaTransferExists.join()) {
                                var aviaBlock = TransportPayload.TransportBlockInfo.copyOf(blockInfo);
                                aviaBlock.setType(TransportPayload.TransportBlockType.AVIA);
                                aviaBlock.setFrom("c" + fromRegion.getGeoId());
                                aviaBlock.setTo("c" + toRegion.getGeoId());
                                blocks.add(aviaBlock);
                            }
                        } catch (CompletionException | CancellationException e) {
                            log.error("Avia block for transport crossSale cannot be fetched", e);
                        }
                        try {
                            if (trainTransferExists.join()) {
                                var trainBlock = TransportPayload.TransportBlockInfo.copyOf(blockInfo);
                                trainBlock.setType(TransportPayload.TransportBlockType.TRAIN);
                                trainSettlementDataProvider.getOptionalSettlementByGeoId(fromRegion.getGeoId())
                                        .ifPresent(tSettlement -> trainBlock.setFrom(tSettlement.getSlug()));
                                trainSettlementDataProvider.getOptionalSettlementByGeoId(toRegion.getGeoId())
                                        .ifPresent(tSettlement -> trainBlock.setTo(tSettlement.getSlug()));
                                blocks.add(trainBlock);
                            }
                        } catch (CompletionException | CancellationException e) {
                            log.error("Train block for transport crossSale cannot be fetched", e);
                        }
                        try {
                            if (busTransferExists.join()) {
                                var busBlock = TransportPayload.TransportBlockInfo.copyOf(blockInfo);
                                busBlock.setType(TransportPayload.TransportBlockType.BUS);
                                trainSettlementDataProvider.getOptionalSettlementByGeoId(fromRegion.getGeoId())
                                        .ifPresent(tSettlement -> busBlock.setFrom(tSettlement.getSlug()));
                                trainSettlementDataProvider.getOptionalSettlementByGeoId(toRegion.getGeoId())
                                        .ifPresent(tSettlement -> busBlock.setTo(tSettlement.getSlug()));
                                blocks.add(busBlock);
                            }
                        } catch (CompletionException | CancellationException e) {
                            log.error("Bus block for transport crossSale cannot be fetched", e);
                        }
                        if (blocks.size() > 0) {
                            transportPayload.setTransportBlocks(blocks);
                        }
                        return blockInfoBuilder
                                .uiPayload(transportPayload)
                                .order(blockSettings.getOrder())
                                .build();
                    }).orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
                }
            case COMMON_PROMO_CODE:
                PromoCodePayload promoCodePayload = new PromoCodePayload();
                // checking for personal promo code or trying to use common static info
                if (protoOrderInfo.getResult().hasGeneratedPromoCodes()) {
                    TGeneratedPromoCodesInfo protoPromoCode = protoOrderInfo.getResult().getGeneratedPromoCodes();
                    promoCodePayload.setPromoCodeName(protoPromoCode.getCode());
                    // TODO(akormushkin): add support for percents
                    promoCodePayload.setDiscountAmount(ProtoUtils.fromTPrice(protoPromoCode.getDiscountAmount()));
                    if (protoPromoCode.hasMinTotalCost()) {
                        promoCodePayload.setMinOrderPrice(ProtoUtils.fromTPrice(protoPromoCode.getMinTotalCost()));
                    } else {
                        promoCodePayload.setMinOrderPrice(Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB));
                    }
                    promoCodePayload.setValidTill(ProtoUtils.toInstant(protoPromoCode.getValidTill()));
                    promoCodePayload.setShowFrom(ProtoUtils.toInstant(protoPromoCode.getValidFrom()));
                    promoCodePayload.setShowTill(ProtoUtils.toInstant(protoPromoCode.getValidTill()).minus(3,
                            ChronoUnit.HOURS));
                    return CompletableFuture.supplyAsync(() -> blockInfoBuilder
                            .uiPayload(promoCodePayload)
                            .order(blockSettings.getOrder())
                            .build())
                            .orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
                } else {
                    HappyPageProperties.PromoCodeSettings promoCodeSettings = config.getPromoCodes().stream()
                            .filter(promoCode -> orderType == promoCode.getForOrderType() &&
                                    promoCode.getShowTill().isAfter(Instant.now()) &&
                                    promoCode.getShowFrom().isBefore(Instant.now()) &&
                                    promoCode.getValidTill().isAfter(Instant.now()))
                            .findFirst().orElse(null);
                    if (promoCodeSettings != null) {
                        promoCodePayload.setPromoCodeName(promoCodeSettings.getName());
                        promoCodePayload.setDiscountAmount(Money.of(promoCodeSettings.getDiscountAmount(),
                                ProtoCurrencyUnit.RUB));
                        promoCodePayload.setMinOrderPrice(Money.of(promoCodeSettings.getMinOrderPrice(),
                                ProtoCurrencyUnit.RUB));
                        promoCodePayload.setValidTill(promoCodeSettings.getValidTill());
                        promoCodePayload.setShowFrom(promoCodeSettings.getShowFrom());
                        promoCodePayload.setShowTill(promoCodeSettings.getShowTill());
                        return CompletableFuture.supplyAsync(() -> blockInfoBuilder
                                .uiPayload(promoCodePayload)
                                .order(blockSettings.getOrder())
                                .build())
                                .orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
                    } else {
                        return CompletableFuture.failedFuture(new RuntimeException("No common promo code found in " +
                                "config"));
                    }
                }
            case AFISHA_CROSS_SALE:
                int destinationGeoId = orderGeoIDExtractor.extract(order);
                var destinationRegion = regionsService.getRegion(destinationGeoId, "ru", "ru");
                if (destinationGeoId == 0) {
                    return CompletableFuture.failedFuture(new RuntimeException("Destination geoId = 0. " +
                            "Afisha carousel will not be shown"));
                }

                CompletableFuture<String> cityUrlFuture = afishaService.getCityUrls(destinationGeoId)
                        .thenApply(AfishaCitiesResponse::getUrl);
                LocalDate reqDate = getAfishaParamsFromOrder(order);
                return afishaService.getActualEvents(destinationGeoId,
                        blockSettings.getResultsLimit(),
                        2,
                        reqDate,
                        AfishaImageSize.SMALL
                ).thenApply(response -> {
                    AfishaCrossSalePayload afishaPayload = new AfishaCrossSalePayload();
                    afishaPayload.setRegionLinguistics(destinationRegion.getLinguistics());
                    afishaPayload.setEvents(afishaResponseMapper.getEventsFromResponse(afishaProperties.getLinkBaseUrl(), reqDate.format(DateTimeFormatter.ISO_DATE), response));
                    afishaPayload.setRegionUrl(afishaProperties.getLinkBaseUrl() + cityUrlFuture.join() + "/events");
                    return blockInfoBuilder
                            .uiPayload(afishaPayload)
                            .order(blockSettings.getOrder())
                            .build();
                }).orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
            case IZI_TRAVEL_CROSS_SALE:
                int destGeoId = orderGeoIDExtractor.extract(order);
                if (destGeoId == 0) {
                    return CompletableFuture.failedFuture(new RuntimeException("Destination geoId = 0. " +
                            "Izi.travel carousel will not be shown"));
                }
                var destRegion = regionsService.getRegion(destGeoId, "en", "en");
                AtomicReference<UUID> cityUuidRef = new AtomicReference<>();
                return iziTravelService.searchCityByNameAndCoords(destRegion.getLinguistics().getNominativeCase(), destRegion.getCoordinates())
                        .thenCompose(cityUuid -> {
                            if (cityUuid == null) {
                                throw new RuntimeException("City not found in IZI.TRAVEL");
                            }
                            cityUuidRef.set(cityUuid);
                            return iziTravelService.getCityChildren(cityUuid, null);
                        })
                        .thenApply(citiesChildrenResponse -> citiesChildrenResponse.stream()
                                .filter(IziTravelHelper::checkCityChildIsAllowed)
                                .map(IziTravelCitiesChildrenResponse::getUuid)
                                .collect(Collectors.toList()))
                        .thenCompose(iziTravelService::getBatchOfObjects)
                        .thenApply(searchResponse -> {
                            IziTravelCrossSalePayload iziPayload = new IziTravelCrossSalePayload();
                            iziPayload.setDirectUrl(iziTravelProperties.getLinkBaseUrl() + "/city/" + cityUuidRef.get().toString());
                            iziPayload.setTours(iziTravelResponseMapper.getToursFromResponse(iziTravelProperties.getLinkBaseUrl(), searchResponse));

                            return blockInfoBuilder
                                    .uiPayload(iziPayload)
                                    .order(blockSettings.getOrder())
                                    .build();
                        }).orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
            case YANDEX_MARKET:
                if (Strings.isNullOrEmpty(fromGeoId)) {
                    return CompletableFuture.failedFuture(new RuntimeException("fromGeoId = null. " +
                            "Market widget will not be shown"));
                }
                return marketWidgetPayloadProvider.get(Integer.parseInt(fromGeoId), blockSettings)
                        .thenApply(marketWidgetPayload -> blockInfoBuilder
                                .uiPayload(marketWidgetPayload)
                                .order(blockSettings.getOrder())
                                .build())
                        .orTimeout(config.getLoadBlockTimeout().toMillis(), TimeUnit.MILLISECONDS);
            default:
                return CompletableFuture.failedFuture(new IllegalArgumentException("Unknown blockType: " + blockSettings.getType()));
        }
    }

    private TransportPayload.TransportBlockInfo getTransportRequestParamsFromOrder(HappyPageOrder order) {
        var blockInfo = new TransportPayload.TransportBlockInfo();
        if (order instanceof HotelHappyPageOrder) {
            var hotelOrder = (HotelHappyPageOrder) order;
            blockInfo.setDepartureAt(hotelOrder.getOrderInfo().getRequestInfo().getCheckinDate().minusDays(1));
            blockInfo.setAdults(hotelOrder.getOrderInfo().getRequestInfo().getNumAdults());
            blockInfo.setChildrenAges(hotelOrder.getOrderInfo().getRequestInfo().getChildAges());
        }
        return blockInfo;
    }

    private LocalDate getAfishaParamsFromOrder(HappyPageOrder order) {
        if (order instanceof HotelHappyPageOrder) {
            var hotelOrder = (HotelHappyPageOrder) order;
            return hotelOrder.getOrderInfo().getRequestInfo().getCheckinDate().plusDays(1);
        } else if (order instanceof TrainHappyPageOrder) {
            var trainOrder = (TrainHappyPageOrder) order;
            return LocalDate.ofInstant(trainOrder.getArrival(), ZoneOffset.UTC).plusDays(1);
        } else if (order instanceof AviaHappyPageOrder) {
            var aviaOrder = (AviaHappyPageOrder) order;
            return OrderFlightsExtractor.getFinalFlightFromForwardSegment(aviaOrder).getArrival().toLocalDate().plusDays(1);
        } else {
            return LocalDate.now();
        }
    }

    private boolean isNotConfirmedAviaOrder(HappyPageOrder order) {
        if (order instanceof AviaHappyPageOrder) {
            var aviaOrder = (AviaHappyPageOrder) order;
            return EDisplayOrderState.OS_FULFILLED != aviaOrder.getEDisplayOrderState();
        } else {
            return false;
        }
    }
}
