package ru.yandex.travel.api.services.orders.happy_page;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.service.dto.FlightDTO;
import ru.yandex.travel.api.models.hotels.RegionSearchHotelsRequestData;
import ru.yandex.travel.api.models.train.Passenger;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.static_pages.RegionHotelsSearcher;
import ru.yandex.travel.api.services.orders.happy_page.model.AviaHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.HappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.HotelPayload;
import ru.yandex.travel.api.services.orders.happy_page.model.PromoPayload;
import ru.yandex.travel.api.services.orders.happy_page.model.TrainHappyPageOrder;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;

@Service
@RequiredArgsConstructor
public class HotelPayloadProvider {
    private final RegionHotelsSearcher regionHotelsSearcher;
    private final RegionHotelsMapper regionHotelsMapper;
    private final RegionsService regionsService;

    public CompletableFuture<HotelPayload> get(HappyPageOrder order,
                                               Integer geoId,
                                               HappyPageProperties.BlockSettings blockSettings,
                                               CommonHttpHeaders headers,
                                               UserCredentials userCredentials,
                                               String logPrefix) {
        var domain = "ru";
        var region = regionsService.getRegion(geoId, domain, "ru");
        var requestData = new RegionSearchHotelsRequestData();
        requestData.setLimit(blockSettings.getResultsLimit());
        return regionHotelsSearcher.searchHotelsWithMinPrices(geoId, requestData, headers, userCredentials, logPrefix)
                .thenApply(searchResult -> {
                    Preconditions.checkState(geoId != 0, "Zero geoId. Carousel will not be shown");
                    Preconditions.checkState(searchResult.getHotelsWithMinPrice().size() > 0,
                            "Hotels not found. Carousel will not be shown");
                    HotelPayload hotelPayload = new HotelPayload();
                    hotelPayload.setTargetCity(region.getLinguistics().getGenitiveCase());
                    hotelPayload.setRegionGeoId(geoId);
                    hotelPayload.setRegionLinguistics(region.getLinguistics());
                    HotelPayload.RequestParams requestParams = getHotelsRequestParamsFromOrder(order);
                    Preconditions.checkState(requestParams.getNights() > 0,
                            "Zero nights set. Carousel will not be shown");
                    hotelPayload.setRequestParams(requestParams);
                    if (blockSettings.isPromoEnabled()) {
                        hotelPayload.setPromoBanner(new PromoPayload(blockSettings.getPromoAdFoxId()));
                    }
                    hotelPayload.setHotelsList(regionHotelsMapper.getHotelsWithMinPrice(requestParams.getNights(), searchResult));
                    return hotelPayload;
                });
    }

    private HotelPayload.RequestParams getHotelsRequestParamsFromOrder(HappyPageOrder order) {
        var requestParams = new HotelPayload.RequestParams();
        if (order instanceof TrainHappyPageOrder) {
            var trainOrder = (TrainHappyPageOrder) order;
            requestParams.setNights(1);
            requestParams.setCheckinDate(LocalDate.ofInstant(trainOrder.getArrival(),
                    ZoneId.of(trainOrder.getStationTo().getTimezone())));
            requestParams.setCheckoutDate(requestParams.getCheckinDate().plusDays(requestParams.getNights()));
            int numAdults = 0;
            List<Integer> childrenAges = new ArrayList<>();
            for (Passenger passenger : trainOrder.getPassengers()) {
                if (passenger.getAge() < 18) {
                    childrenAges.add(passenger.getAge());
                } else {
                    numAdults = numAdults + 1;
                }
            }
            requestParams.setAdults(numAdults);
            if (childrenAges.size() > 0) {
                requestParams.setChildrenAges(childrenAges);
            }
            return requestParams;
        } else if (order instanceof AviaHappyPageOrder) {
            var aviaOrder = (AviaHappyPageOrder) order;
            FlightDTO forwardFlight = OrderFlightsExtractor.getFinalFlightFromForwardSegment(aviaOrder);
            LocalDate dateFrom = forwardFlight.getArrival().toLocalDate();
            if (forwardFlight.getArrival().getHour() < 6) {
                dateFrom = dateFrom.minusDays(1);
            }
            requestParams.setCheckinDate(dateFrom);
            if (aviaOrder.getAirReservation().getSegments().size() == 2) {
                FlightDTO backwardFlight = OrderFlightsExtractor.getFirstFlightFromBackwardSegment(aviaOrder);
                requestParams.setCheckoutDate(backwardFlight.getDeparture().toLocalDate());
                int nights = (int) (requestParams.getCheckoutDate().toEpochDay() - requestParams.getCheckinDate().toEpochDay());
                requestParams.setNights(nights);
            } else {
                requestParams.setNights(1);
                requestParams.setCheckoutDate(requestParams.getCheckinDate().plusDays(requestParams.getNights()));
            }
            int adults = 0;
            List<Integer> childrenAges = new ArrayList<>();
            for (var travellerDTO : aviaOrder.getTravellers()) {
                switch (travellerDTO.getCategory()) {
                    case ADULT:
                        adults = adults + 1;
                        break;
                    case CHILD:
                    case INFANT:
                        childrenAges.add(countPassengerAge(travellerDTO.getDateOfBirth()));
                        break;
                    default:
                        break;
                }
            }
            requestParams.setAdults(adults);
            requestParams.setChildrenAges(childrenAges);
            return requestParams;
        } else {
            return requestParams;
        }
    }

    private int countPassengerAge(LocalDate passengerBirthday) {
        if (passengerBirthday == null) {
            return 0;
        }
        LocalDate today = LocalDate.now();
        int age = today.getYear() - passengerBirthday.getYear();
        if (today.getDayOfYear() < passengerBirthday.getDayOfYear()) {
            age -= 1;
        }
        return age;
    }
}
