package ru.yandex.travel.api.services.orders.happy_page;

import java.util.concurrent.CompletableFuture;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.orders.happy_page.model.MarketWidgetCrossSalePayload;

@Service
@RequiredArgsConstructor
@EnableConfigurationProperties(MarketWidgetProperties.class)
public class MarketWidgetPayloadProvider {
    private final RegionsService regionsService;
    private final MarketWidgetProperties config;

    public CompletableFuture<MarketWidgetCrossSalePayload> get(Integer fromGeoId,
                                                               HappyPageProperties.BlockSettings blockSettings
                                                               ) {
        String domain = "ru";
        if (config.getMarketWidgetSettings() == null) {
            return CompletableFuture.failedFuture(new RuntimeException("Market Widget settings are not configured"));
        }
        if (!config.getMarketWidgetSettings().containsKey(blockSettings.getMarketWidgetName())) {
            return CompletableFuture.failedFuture(
                    new RuntimeException("There is no Market Widget config with name " + blockSettings.getMarketWidgetName())
            );
        }
        MarketWidgetProperties.MarketWidgetSettings marketWidgetSettings = config.getMarketWidgetSettings()
                .get(blockSettings.getMarketWidgetName());
        MarketWidgetProperties.MarketWidgetFilters marketWidgetFilters = marketWidgetSettings.getFilters();

        if (marketWidgetFilters != null) {
            if (marketWidgetFilters.getRegionGeoId() != null) {
                Integer fromRegionGeoId = regionsService.getRegionRoundTo(
                        fromGeoId, GeoBaseHelpers.REGION_REGION_TYPE, domain
                );
                if (!marketWidgetFilters.getRegionGeoId().equals(fromRegionGeoId)) {
                    return CompletableFuture.failedFuture(
                            new RuntimeException("Wrong fromRegionGeoId for this Market widget")
                    );
                }
            }
            if (marketWidgetFilters.getCountryGeoId() != null) {
                Integer fromCountryGeoId = regionsService.getRegionRoundTo(
                        fromGeoId, GeoBaseHelpers.COUNTRY_REGION_TYPE, domain
                );
                if (!marketWidgetFilters.getCountryGeoId().equals(fromCountryGeoId)) {
                    return CompletableFuture.failedFuture(
                            new RuntimeException("Wrong fromCountryGeoId for this Market widget")
                    );
                }
            }
        }

        return CompletableFuture.supplyAsync(() -> {
            MarketWidgetCrossSalePayload marketWidgetPayload = new MarketWidgetCrossSalePayload();
            marketWidgetPayload.setTitle(marketWidgetSettings.getTitle());
            marketWidgetPayload.setType(marketWidgetSettings.getType());
            marketWidgetPayload.setFallback(marketWidgetSettings.getFallback());
            marketWidgetPayload.setParams(buildMarketWidgetParameters(marketWidgetSettings.getParams()));
            return marketWidgetPayload;
        });
    }

    private MarketWidgetCrossSalePayload.MarketWidgetParameters buildMarketWidgetParameters(
            MarketWidgetProperties.MarketWidgetParameters configParameters
    ) {
        var payloadParameters = new MarketWidgetCrossSalePayload.MarketWidgetParameters();
        payloadParameters.setClid(configParameters.getClid());
        payloadParameters.setThemeId(configParameters.getThemeId());
        payloadParameters.setSearchModelIds(configParameters.getSearchModelIds());
        payloadParameters.setButtonLinkType(configParameters.getButtonLinkType());
        payloadParameters.setThemeRows(configParameters.getThemeRows());
        payloadParameters.setMetrikaCounterId(configParameters.getMetrikaCounterId());
        return payloadParameters;
    }
}
