package ru.yandex.travel.api.services.orders.happy_page;

import java.util.List;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.models.train.PlaceWithType;
import ru.yandex.travel.api.services.orders.happy_page.model.HappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.HotelHappyPageOrder;
import ru.yandex.travel.api.services.orders.happy_page.model.TrainHappyPageOrder;
import ru.yandex.travel.api.services.orders.notifier.model.OrderInfoPayload;

@Service
@RequiredArgsConstructor
public class OrderInfoPayloadExtractor {
    public OrderInfoPayload extract(HappyPageOrder order) {
        OrderInfoPayload result = new OrderInfoPayload();
        if (order instanceof TrainHappyPageOrder) {
            TrainHappyPageOrder trainOrder = (TrainHappyPageOrder)order;
            result.setOrderId(trainOrder.getId());
            result.setPrettyId(trainOrder.getPrettyId());
            result.setTrainOrderInfos(extractTrainOrderInfos(trainOrder));
        } else if (order instanceof HotelHappyPageOrder) {
            HotelHappyPageOrder hotelOrder = (HotelHappyPageOrder)order;
            result.setOrderId(hotelOrder.getId());
            result.setPrettyId(hotelOrder.getYandexOrderId());
            result.setHotelOrderInfos(extractHotelOrderInfos(hotelOrder));
        }
        return result;
    }

    private List<OrderInfoPayload.TrainOrderInfo> extractTrainOrderInfos(TrainHappyPageOrder order) {
        OrderInfoPayload.TrainOrderInfo orderInfo = new OrderInfoPayload.TrainOrderInfo();
        if (order.getStationFrom() != null) {
            orderInfo.setStationFromId(Integer.toString(order.getStationFrom().getId()));
        }
        if (order.getStationTo() != null) {
            orderInfo.setStationToId(Integer.toString(order.getStationTo().getId()));
        }
        var trainInfo = order.getTrainInfo();
        if (trainInfo != null) {
            orderInfo.setTrainStartSettlementTitle(trainInfo.getStartSettlementTitle());
            orderInfo.setTrainEndSettlementTitle(trainInfo.getEndSettlementTitle());
            orderInfo.setTrainNumber(trainInfo.getTrainNumber());
            orderInfo.setBrandTitle(trainInfo.getBrandTitle());
        }
        orderInfo.setDeparture(order.getDeparture());
        orderInfo.setArrival(order.getArrival());
        orderInfo.setCarNumber(order.getCarNumber());
        orderInfo.setCarType(order.getCarType());
        orderInfo.setCompartmentGender(order.getCompartmentGender());
        orderInfo.setPlaceNumbers(
                order.getPassengers().stream()
                        .flatMap(p -> p.getTickets().stream())
                        .flatMap(t -> t.getPlaces().stream())
                        .map(PlaceWithType::getNumber)
                        .distinct()
                        .sorted()
                        .collect(Collectors.toList()));
        return List.of(orderInfo);
    }

    private List<OrderInfoPayload.HotelOrderInfo> extractHotelOrderInfos(HotelHappyPageOrder order) {
        OrderInfoPayload.HotelOrderInfo orderInfo = new OrderInfoPayload.HotelOrderInfo();
        var basicHotelInfo = order.getOrderInfo().getBasicHotelInfo();
        if (basicHotelInfo != null) {
            orderInfo.setName(basicHotelInfo.getName());
            orderInfo.setAddress(basicHotelInfo.getAddress());
            orderInfo.setPhone(basicHotelInfo.getPhone());
            orderInfo.setImageUrlTemplate(basicHotelInfo.getImageUrlTemplate());
        }
        var partnerHotelInfo = order.getOrderInfo().getPartnerHotelInfo();
        if (partnerHotelInfo != null) {
            if (partnerHotelInfo.getCheckin() != null) {
                orderInfo.setCheckInBeginTime(partnerHotelInfo.getCheckin().getBeginTime());
                orderInfo.setCheckInEndTime(partnerHotelInfo.getCheckin().getEndTime());
            }
            if (partnerHotelInfo.getCheckout() != null) {
                orderInfo.setCheckOutTime(partnerHotelInfo.getCheckout().getTime());
            }
        }
        if (order.getConfirmationInfo() != null && order.getConfirmationInfo().getDocumentUrl() != null) {
            orderInfo.setDocumentUrl(order.getConfirmationInfo().getDocumentUrl());
        }
        var requestInfo = order.getOrderInfo().getRequestInfo();
        if (requestInfo != null) {
            if (requestInfo.getCheckinDate() != null) {
                orderInfo.setCheckInDate(requestInfo.getCheckinDate());
            }
            if (requestInfo.getCheckoutDate() != null) {
                orderInfo.setCheckOutDate(requestInfo.getCheckoutDate());
            }
        }
        return List.of(orderInfo);
    }
}
