package ru.yandex.travel.api.services.orders.happy_page.afisha;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaActualEventsResponse;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaCitiesResponse;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaCityInfoResponse;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaGraphQLRequestData;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaGraphQLResponse;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.AfishaImageSize;
import ru.yandex.travel.api.services.orders.happy_page.afisha.model.GraphQLRequestVariables;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.HttpApiClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableHttpException;
import ru.yandex.travel.tvm.TvmWrapper;

@Service
@Slf4j
@EnableConfigurationProperties(AfishaServiceProperties.class)
public class AfishaService extends HttpApiClientBase {
    public static final String CITIES_QUERY = "{ cities { name geoid url } }";
    public static final String CITY_INFO_QUERY = "{ cityInfo { name geoid url coordinates {latitude longitude} } }";
    public static final String ACTUAL_EVENTS_QUERY = "query " +
            "HotelsHappyPageActualEvents($limit: PagingLimitInput!, $date: Date!, $period: Int!, $size: MediaImageSizes!) {" +
            "  actualEvents(" +
            "    paging: { limit: $limit, offset: 0 }" +
            "    dates: { date: $date, period: $period }" +
            "  ) {" +
            "    items {" +
            "      scheduleInfo {" +
            "        collapsedText" +
            "        preview {" +
            "          text" +
            "        }" +
            "        regularity {" +
            "          singleShowtime" +
            "        }" +
            "      }" +
            "      event {" +
            "        id" +
            "        title" +
            "        contentRating" +
            "        url" +
            "        tickets {" +
            "          minPrice {" +
            "            currency" +
            "            value" +
            "          }" +
            "        }" +
            "        type {" +
            "          name" +
            "        }" +
            "        tags(status: approved){" +
            "          name" +
            "        }" +
            "        image {" +
            "          image(size: $size) {" + //s600x314_wmark_tickets
            "            url" +
            "          }" +
            "        }" +
            "      }" +
            "    }" +
            "  }" +
            "}";

    private final TvmWrapper tvm;
    private final String tvmAlias;
    private final boolean tvmEnabled;

    public AfishaService(@Qualifier("afishaAhcWrapper") AsyncHttpClientWrapper ahcClient,
                         AfishaServiceProperties config,
                         @Autowired(required = false) TvmWrapper tvm
                         ) {
        super(ahcClient, config, new ObjectMapper()
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .registerModule(new JavaTimeModule()));
        this.tvm = tvm;
        this.tvmAlias = config.getTvmAlias();
        this.tvmEnabled = config.getTvmEnabled();
        if (tvmEnabled && tvm != null) {
            tvm.validateAlias(tvmAlias);
        }
    }

    public CompletableFuture<AfishaCityInfoResponse> getClosestCityInfo(Integer geoId) {
        return executeGraphQLRequest(geoId, CITY_INFO_QUERY, null)
                .thenApply(response -> {
                    int status = response.getStatus();
                    if (status == 200) {
                        if (response.getData() != null) {
                            return response.getData().getCityInfo();
                        } else {
                            throw new RuntimeException("Afisha::cityInfo returned non-200: " + response.getQueryErrors().toString());
                        }
                    } else if (500 <= status) {
                        throw new RetryableHttpException(status, response.getQueryErrors().toString());
                    } else {
                        String reason = response.getQueryErrors().toString();
                        throw new RuntimeException("Afisha::cityInfo request failed with code: " + status + ". " +
                                "Reason: " + reason);
                    }
                });
    }

    public CompletableFuture<AfishaCitiesResponse> getCityUrls(Integer geoId) {
        return executeGraphQLRequest(geoId, CITIES_QUERY, null)
                .thenApply(response -> {
                    int status = response.getStatus();
                    if (status == 200) {
                        if (response.getData() != null) {
                            return response.getData().getCities().stream()
                                    .filter(cityInfo -> geoId.equals(cityInfo.getGeoid()))
                                    .findFirst()
                                    .orElseThrow();
                        } else {
                            throw new RuntimeException("Afisha::cities returned non-200: " + response.getQueryErrors().getCities().toString());
                        }
                    } else if (500 <= status) {
                        throw new RetryableHttpException(status, response.getQueryErrors().getCities().toString());
                    } else {
                        String reason = response.getQueryErrors().getCities().getMessage();
                        throw new RuntimeException("Afisha::cities request failed with code: " + status + ". " +
                                "Reason: " + reason);
                    }
                });
    }

    public CompletableFuture<AfishaActualEventsResponse> getActualEvents(Integer geoId,
                                                                         Integer limit,
                                                                         Integer period,
                                                                         LocalDate date,
                                                                         AfishaImageSize imageSize) {
        ActualEventsVariables actualEventsVariables = new ActualEventsVariables();
        actualEventsVariables.setLimit(limit);
        actualEventsVariables.setDate(date.format(DateTimeFormatter.ISO_DATE));
        actualEventsVariables.setPeriod(period);
        actualEventsVariables.setSize(imageSize.getValue());

        return executeGraphQLRequest(geoId, ACTUAL_EVENTS_QUERY, actualEventsVariables)
                .thenApply(response -> {
                    int status = response.getStatus();
                    if (status == 200) {
                        if (response.getData() != null) {
                            return response.getData().getActualEvents();
                        } else {
                            throw new RuntimeException("Afisha::actualEvents returned non-200: " + response.getQueryErrors().getActualEvents().toString());
                        }
                    } else if (500 <= status) {
                        throw new RetryableHttpException(status, response.getQueryErrors().getActualEvents().toString());
                    } else {
                        String reason = response.getQueryErrors().getActualEvents().getMessage();
                        throw new RuntimeException("Afisha::actualEvents request failed with code: " + status + ". " +
                                "Reason: " + reason);
                    }
                });
    }

    public CompletableFuture<AfishaGraphQLResponse> executeGraphQLRequest(Integer geoId, String query, GraphQLRequestVariables variables) {
        AfishaGraphQLRequestData requestData = new AfishaGraphQLRequestData();
        requestData.setGeoid(geoId);
        requestData.setQuery(query);
        if (variables != null) {
            requestData.setVariables(variables);
        }
        requestData.setLang("ru");

        return sendRequest(
                "POST",
                "?query_name=HotelsHappyPage",
                requestData,
                AfishaGraphQLResponse.class,
                "graphQL"
        );
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        RequestBuilder rb = super.createBaseRequestBuilder(method, path, body);
        if (tvmEnabled && tvm != null) {
            rb.addHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), tvm.getServiceTicket(tvmAlias));
        }
        rb.addHeader("Content-Type", "application/json");
        return rb;
    }

    @EqualsAndHashCode(callSuper = true)
    @ToString(callSuper = true)
    @Data
    public static class ActualEventsVariables extends GraphQLRequestVariables{
        private Integer limit;
        private String date;
        private Integer period;
        private String size;
    }
}
