package ru.yandex.travel.api.services.orders.happy_page.izi_travel;

import java.time.Duration;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import org.asynchttpclient.RequestBuilder;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.misc.geo.Coordinates;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelCitiesChildrenResponse;
import ru.yandex.travel.api.services.orders.happy_page.izi_travel.model.IziTravelSearchResponse;
import ru.yandex.travel.commons.http.apiclient.HttpApiClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpApiRetryStrategy;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.Retry;

@Service
@EnableConfigurationProperties(IziTravelProperties.class)
public class IziTravelService extends HttpApiClientBase {
    private static final String IZI_API_KEY_HEADER = "X-IZI-API-KEY";
    private final String apiToken;
    private final Retry retryHelper;
    private final HttpApiRetryStrategy retryStrategy;

    public IziTravelService(@Qualifier("iziTravelAhcWrapper") AsyncHttpClientWrapper asyncHttpClient,
                            IziTravelProperties config,
                            ObjectMapper objectMapper,
                            Retry retry) {
        super(asyncHttpClient, config, objectMapper);
        this.apiToken = config.getToken();
        this.retryHelper = retry;
        this.retryStrategy = new HttpApiRetryStrategy(Duration.ZERO, config.getTriesCount(),
                HttpApiRetryStrategy.DEFAULT_RETRYABLE_EXCEPTIONS);
    }

    public CompletableFuture<UUID> searchCityByNameAndCoords(String name, Coordinates coordinates) {
        return search(coordinates, 100000L, name, 1, "city")
                .thenApply(response -> {
                    if (response != null && response.size() > 0) {
                        return response.get(0).getUuid();
                    } else {
                        return null;
                    }
                });
    }

    protected <RQ, RS> CompletableFuture<RS> sendRequestWithRetry(String method, String path, RQ body,
                                                                  Class<RS> responseType, String purpose) {
        return retryHelper.withRetry(
                purpose,
                () -> this.sendRequest(method, path, body, responseType, purpose),
                this.retryStrategy);
    }

    public CompletableFuture<List<IziTravelCitiesChildrenResponse>> getCityChildren(UUID cityUuid, Integer limit) {
        String path = "/cities/" + cityUuid + "/children?media_links=true&languages=ru&type=tour,museum";
        if (limit != null) {
            path = path + "&limit=" + limit;
        }
        return sendRequestWithRetry("GET", path, null, IziTravelCitiesChildrenResponse[].class, "IziTravelCity")
                .thenApply(List::of);
    }

    public CompletableFuture<List<IziTravelSearchResponse>> getBatchOfObjects(List<UUID> objectUuidList) {
        String path = "/mtgobjects/batch/" +
                objectUuidList.stream().map(UUID::toString).collect(Collectors.joining(",")) +
                "?languages=ru&media_links=true";
        return sendRequestWithRetry("GET", path, null, IziTravelSearchResponse[].class, "IziTravelBatchGetObject")
                .thenApply(List::of);
    }

    public CompletableFuture<List<IziTravelSearchResponse>> search(Coordinates coordinates, Long radius, String query
            , Integer limit, String type) {
        StringBuilder pathBuilder = new StringBuilder("/mtg/objects/search?version=1.8&languages=ru&media_links=true");
        // https://api-docs.izi.travel/#search-for-mtgobjects-cities-and-countries

        if (coordinates != null && radius != null) {
            pathBuilder.append("&lat_lon=").append(coordinates.getLatitude()).append(",").append(coordinates.getLongitude());
            pathBuilder.append("&radius=").append(radius);
        }
        if (!Strings.isNullOrEmpty(query)) {
            pathBuilder.append("&query=").append(query);
        }
        if (!Strings.isNullOrEmpty(type)) {
            pathBuilder.append("&type=").append(type);
        }
        pathBuilder.append("&limit=").append(limit);
        return sendRequestWithRetry("GET", pathBuilder.toString(), null, IziTravelSearchResponse[].class,
                "IziTravelSearch")
                .thenApply(List::of);
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        RequestBuilder rb = super.createBaseRequestBuilder(method, path, body);
        if (!Strings.isNullOrEmpty(apiToken)) {
            rb.addHeader(IZI_API_KEY_HEADER, apiToken);
        }
//        rb.addHeader("Accept", "application/izi-api-v1.8+json");
        return rb;
    }
}
