package ru.yandex.travel.api.services.orders.happy_page.morda_backend;

import java.io.IOException;
import java.time.LocalDate;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.asynchttpclient.util.HttpConstants;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.AhcHttpRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;

@Service
@Slf4j
@EnableConfigurationProperties(MordaBackendServiceProperties.class)
public class MordaBackendService {
    private final AsyncHttpClientWrapper ahcClient;
    private final MordaBackendServiceProperties config;
    private final ObjectMapper objectMapper;
    private final Retry retryHelper;
    private final TrainSettlementDataProvider trainSettlementDataProvider;

    public MordaBackendService(@Qualifier("mordaBackendAhcWrapper") AsyncHttpClientWrapper ahcClient,
                               MordaBackendServiceProperties config,
                               Retry retry,
                               TrainSettlementDataProvider trainSettlementDataProvider) {
        this.ahcClient = ahcClient;
        this.config = config;
        this.retryHelper = retry;
        this.trainSettlementDataProvider = trainSettlementDataProvider;
        this.objectMapper = new ObjectMapper()
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .registerModule(new JavaTimeModule());
    }

    public CompletableFuture<Boolean> checkTrainTransferExists(int geoIdFrom, int geoIdTo, LocalDate when) {
        Optional<TSettlement> settlementFrom = trainSettlementDataProvider.getOptionalSettlementByGeoId(geoIdFrom);
        Optional<TSettlement> settlementTo = trainSettlementDataProvider.getOptionalSettlementByGeoId(geoIdTo);
        if (settlementFrom.isEmpty() || settlementTo.isEmpty()) {
            log.error("Could not get settlements from geoId for HappyPage Transport blocks");
            return CompletableFuture.completedFuture(false);
        }
        String raspIdFrom = TrainSettlementDataProvider.Companion.prepareRaspSettlementId(settlementFrom.get().getId());
        String raspIdTo = TrainSettlementDataProvider.Companion.prepareRaspSettlementId(settlementTo.get().getId());
        return search(raspIdFrom, raspIdTo, "train", when)
                .thenApply(response ->
                        response.getResult() != null &&
                                response.getResult().getSegments() != null &&
                                response.getResult().getSegments().size() > 0
                );
    }

    public CompletableFuture<SearchResponse> search(
            String pointFrom,
            String pointTo,
            String transportType,
            LocalDate when
    ) {
        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod(HttpConstants.Methods.GET)
                .setUrl(config.getBaseUrl() + config.getSearchPath())
                .addQueryParam("transportType", transportType)
                .addQueryParam("pointFrom", pointFrom)
                .addQueryParam("pointTo", pointTo)
                .addQueryParam("when", when.toString());
        return retryHelper.withRetry(
                "MordaBackendService::search/",
                this::runRequest,
                requestBuilder,
                new AhcHttpRetryStrategy()
        ).thenApply(this::parseSearchResponse);
    }

    private CompletableFuture<Response> runRequest(RequestBuilder request) {
        return ahcClient.executeRequest(request);
    }

    private SearchResponse parseSearchResponse(Response response) {
        if (response.getStatusCode() != 200) {
            log.error("MordaBackend::SearchResponse returned non-200 status code: {}", response.getStatusCode());
            return new SearchResponse();
        }
        try {
            return objectMapper.readValue(response.getResponseBody(), SearchResponse.class);
        } catch (IOException e) {
            log.error(response.getResponseBody());
            log.error("Unable to parse response", e);
            throw new RuntimeException("Unable to parse response", e);
        }
    }
}
