package ru.yandex.travel.api.services.orders.suburban;

import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Strings;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.generic_booking_flow.model.suburban.AeroexpressBookData;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.suburban.AeroexpressServiceInfoDTO;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.suburban.CreateSuburbanServiceData;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.suburban.ImBookData;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.suburban.MovistaBookData;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.suburban.SuburbanServiceInfoDTO;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.services.dictionaries.train.station.TrainStationDataProvider;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.TSuburbanTestContext;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TServiceInfo;
import ru.yandex.travel.suburban.model.AeroexpressReservation;
import ru.yandex.travel.suburban.model.ImReservation;
import ru.yandex.travel.suburban.model.MovistaReservation;
import ru.yandex.travel.suburban.model.SuburbanDicts;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.suburban.partners.SuburbanProvider;

@Slf4j
@Service
@AllArgsConstructor
public class SuburbanHelper {
    private final TrainStationDataProvider stationDataProvider;
    private final ApiTokenEncrypter apiTokenEncrypter;

    public List<TCreateServiceReq> buildSuburbanServicesReqs(List<CreateSuburbanServiceData> suburbanServices) {
        List<TCreateServiceReq> servicesReqs = new ArrayList<>();
        if (suburbanServices == null) {
            return servicesReqs;
        }

        for (CreateSuburbanServiceData suburbanService : suburbanServices) {
            // Пока не работает валидация запросов Travel Api, необходимо проверять руками
            SuburbanProvider provider = suburbanService.getProvider();
            if (provider == null) {
                provider = SuburbanProvider.MOVISTA;
            }

            String stationFromTitle = stationDataProvider.getById(suburbanService.getStationFromId()).getTitleDefault();
            String stationToTitle = stationDataProvider.getById(suburbanService.getStationToId()).getTitleDefault();

            // common fields between old and new version
            SuburbanReservation.SuburbanReservationBuilder payload = SuburbanReservation.builder()
                    .provider(provider)
                    .carrier(suburbanService.getCarrierPartner())
                    .stationFrom(SuburbanReservation.Station.builder()
                            .id(suburbanService.getStationFromId())
                            .titleDefault(stationFromTitle).build())
                    .stationTo(SuburbanReservation.Station.builder()
                            .id(suburbanService.getStationToId())
                            .titleDefault(stationToTitle).build())
                    .price(Money.of(suburbanService.getPrice(), ProtoCurrencyUnit.RUB));

            if (provider == SuburbanProvider.MOVISTA) {
                MovistaBookData movistaBookData = suburbanService.getMovistaBookData();
                payload.movistaReservation(MovistaReservation.builder()
                        .date(movistaBookData.getDate())
                        .stationFromExpressId(movistaBookData.getStationFromExpressId())
                        .stationToExpressId(movistaBookData.getStationToExpressId())
                        .fareId(movistaBookData.getFareId())
                        .wicket(movistaBookData.getWicket())
                        .build());
            } else if (provider == SuburbanProvider.IM) {
                ImBookData imBookData = suburbanService.getImBookData();
                payload.imReservation(ImReservation.builder()
                        .date(imBookData.getDate())
                        .stationFromExpressId(imBookData.getStationFromExpressId())
                        .stationToExpressId(imBookData.getStationToExpressId())
                        .trainNumber(imBookData.getTrainNumber())
                        .imProvider(imBookData.getImProvider())
                        .build());
            } else if (provider == SuburbanProvider.AEROEXPRESS) {
                AeroexpressBookData aeroexpressBookData = suburbanService.getAeroexpressBookData();
                payload.aeroexpressReservation(AeroexpressReservation.builder()
                        .menuId(aeroexpressBookData.getMenuId())
                        .orderType(aeroexpressBookData.getOrderType())
                        .date(aeroexpressBookData.getDate())
                        .build());
            } else {
                throw new RuntimeException(String.format("Provider %s is not supported", provider));
            }

            fillSuburbanDicts(payload, suburbanService);
            TJson payloadJson = ProtoUtils.toTJson(payload.build());
            log.info("Creating suburban service with payload: {}", payloadJson.getValue());

            var serviceReq = TCreateServiceReq.newBuilder()
                    .setServiceType(EServiceType.PT_SUBURBAN)
                    .setSourcePayload(payloadJson);

            if (!Strings.isNullOrEmpty(suburbanService.getTestContextToken())) {
                TSuburbanTestContext testContext = apiTokenEncrypter.fromSuburbanTestContextToken(
                        suburbanService.getTestContextToken());
                serviceReq.setSuburbanTestContext(testContext);
            }

            servicesReqs.add(serviceReq.build());
        }

        return servicesReqs;
    }

    private static void fillSuburbanDicts(
            SuburbanReservation.SuburbanReservationBuilder payload, CreateSuburbanServiceData suburbanService) {

        payload.suburbanDicts(SuburbanDicts.builder()
                .stations(List.of(
                        SuburbanDicts.Station.builder().raspID(suburbanService.getStationFromId()).build(),
                        SuburbanDicts.Station.builder().raspID(suburbanService.getStationToId()).build()
                ))
                .build());
    }

    public SuburbanServiceInfoDTO buildSuburbanServiceInfo(TOrderServiceInfo orderServiceInfo) {
        TServiceInfo serviceInfo = orderServiceInfo.getServiceInfo();
        var payload = ProtoUtils.fromTJson(serviceInfo.getPayload(), SuburbanReservation.class);

        AeroexpressReservation aeroexpressReservation = payload.getAeroexpressReservation();
        AeroexpressServiceInfoDTO aeroexpressInfo = aeroexpressReservation != null
                ? new AeroexpressServiceInfoDTO(aeroexpressReservation)
                : null;

        return SuburbanServiceInfoDTO.builder()
                .provider(payload.getProvider())
                .carrierPartner(payload.getCarrier())
                .flow(payload.getSuburbanTicketFlow())
                .barcodePreset(payload.getSuburbanTicketBarcodePreset())
                .date(payload.getDepartureDateTime().toLocalDateTime())
                .stationFromId(payload.getStationFrom().getId())
                .stationToId(payload.getStationTo().getId())
                .price(payload.getPrice().getNumberStripped())
                .partnerOrderId(payload.getProviderOrderId())
                .ticketNumber(payload.getProviderTicketNumber())
                .ticketBody(payload.getProviderTicketBody())
                .wicket(payload.getWicket())
                .aeroexpressInfo(aeroexpressInfo)
                .error(SuburbanServiceInfoDTO.Error.fromSuburbanReservationError(payload.getError()))
                .build();
    }
}
