package ru.yandex.travel.api.services.orders.trips;

import java.time.Instant;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.travel.api.endpoints.booking_flow.DtoMapper;
import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.api.endpoints.booking_flow.model.FulfillmentType;
import ru.yandex.travel.api.endpoints.booking_flow.model.GeoHotelInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.Guest;
import ru.yandex.travel.api.endpoints.booking_flow.model.HotelOrderDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderStatus;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.models.trips.orders.AviaOrderItem;
import ru.yandex.travel.api.models.trips.orders.BusOrderItem;
import ru.yandex.travel.api.models.trips.orders.HotelOrderItem;
import ru.yandex.travel.api.models.trips.orders.HotelOrderListItemPayment;
import ru.yandex.travel.api.models.trips.orders.OrderItem;
import ru.yandex.travel.api.models.trips.orders.TrainOrderItem;
import ru.yandex.travel.api.models.trips.orders.buses.Ride;
import ru.yandex.travel.api.services.avia.orders.AviaOrchestratorModelConverter;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels_booking_flow.HotelOrdersService;
import ru.yandex.travel.api.services.hotels_booking_flow.models.HotelOrder;
import ru.yandex.travel.api.services.orders.OrderListSource;
import ru.yandex.travel.api.services.orders.OrderType;
import ru.yandex.travel.api.services.orders.TrainModelMapService;
import ru.yandex.travel.api.services.orders.model.TrainOrderListInfo;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;

@Slf4j
@Service
@RequiredArgsConstructor
public class OrderItemMapper {
    private final ApiTokenEncrypter apiTokenEncrypter;
    private final GeoBase geoBase;
    private final HotelOrdersService hotelOrdersService;
    private final DtoMapper dtoMapper;
    private final AviaOrchestratorModelConverter aviaOrchestratorModelConverter;
    private final TrainModelMapService trainModelMapService;
    private final Counter orderMappingErrorsCounter =
            Counter.builder("trips.orderMapping.error").tags().register(Metrics.globalRegistry);
    private final Map<OrderStatus, FulfillmentType> hotelsFulfillmentMapping = new ImmutableMap.Builder<OrderStatus,
            FulfillmentType>()
            .put(OrderStatus.CONFIRMED, FulfillmentType.COMPLETED)
            .put(OrderStatus.REFUNDED, FulfillmentType.CANCELLED)
            .build();

    private static final String domain = "ru";

    public OrderItem map(TOrderInfo orderInfo) {
        try {
            return mapInternal(orderInfo);
        } catch (Exception e) {
            log.error("failed to map order item for order id {}", orderInfo.getOrderId());
            orderMappingErrorsCounter.increment();
            return null;
        }
    }

    private OrderItem mapInternal(TOrderInfo orderInfo) {
        var item = new OrderItem();
        switch (orderInfo.getOrderType()) {
            case OT_HOTEL_EXPEDIA:
                item.setHotelItem(mapHotelOrder(orderInfo));
                return item;
            case OT_AVIA_AEROFLOT:
                item.setAviaItem(mapAviaOrder(orderInfo));
                return item;
            case OT_TRAIN:
                item.setTrainItem(mapTrainOrder(orderInfo));
                return item;
            case OT_GENERIC:
                //noinspection SwitchStatementWithTooFewBranches
                switch (orderInfo.getDisplayOrderType()) {
                    case DT_TRAIN:
                        item.setTrainItem(mapTrainOrder(orderInfo));
                        return item;
                    case DT_BUS:
                        item.setBusItem(mapBusOrder(orderInfo));
                    default:
                        log.info("Unsupported generic order display type: {}", orderInfo.getDisplayOrderType());
                        return item;
                }
            default:
                throw new RuntimeException("Unknown order type " + orderInfo.getOrderType());
        }
    }

    private BusOrderItem mapBusOrder(TOrderInfo orderInfo) {
        var item = new BusOrderItem();
        item.setId(orderInfo.getOrderId());
        item.setPassportId(orderInfo.getOwner().getPassportId());
        item.setYandexOrderId(orderInfo.getPrettyId());
        item.setOrderType(OrderType.BUS);
        item.setStatus(DisplayableOrderStatus.fromProto(orderInfo.getDisplayOrderState()));
        item.setState(orderInfo.getDisplayOrderState());
        item.setServicedAt(ProtoUtils.toLocalDateTime(orderInfo.getServicedAt()));
        item.setSource(OrderListSource.ORCHESTRATOR);
        var rides = orderInfo.getServiceList().stream()
                .map(s -> mapServiceToRide(s, orderInfo.getOrderId(), orderInfo.getDocumentUrl()))
                .collect(Collectors.toList());
        item.setRides(rides);
        return item;
    }

    private Ride mapServiceToRide(TOrderServiceInfo protoServiceInfo, String orderId, String documentUrl) {
        var busReservation = ProtoUtils.fromTJson(protoServiceInfo.getServiceInfo().getPayload(), BusReservation.class);
        var ride = busReservation.getRide();
        var result = Ride.builder()
                .rideId(ride.getRideId())
                .supplierId(ride.getSupplierId())
                .busPartner(ride.getBusPartner())
                .supplier(ride.getSupplier())
                .carrier(ride.getCarrier())
                .carrierCode(ride.getCarrierCode())
                .localDepartureTime(ride.getLocalDepartureTime())
                .localArrivalTime(ride.getLocalArrivalTime())
                .duration(ride.getDuration())
                .pointFrom(ride.getPointFrom())
                .titlePointFrom(ride.getTitlePointFrom())
                .pointTo(ride.getPointTo())
                .titlePointTo(ride.getTitlePointTo())
                .bus(ride.getBus())
                .routeName(ride.getRouteName())
                .routeNumber(ride.getRouteNumber())
                .refundedTicketsCount((int) busReservation.getOrder().getTickets().stream().filter(t -> t.getStatus() == BusTicketStatus.RETURNED).count());

        if ((protoServiceInfo.getServiceInfo().getGenericOrderItemState() == EOrderItemState.IS_CONFIRMED ||
                protoServiceInfo.getServiceInfo().getGenericOrderItemState() == EOrderItemState.IS_REFUNDING ||
                protoServiceInfo.getServiceInfo().getGenericOrderItemState() == EOrderItemState.IS_REFUNDED) &&
                !Strings.isNullOrEmpty(documentUrl)) {
            TDownloadBlankToken token = TDownloadBlankToken.newBuilder().setOrderId(orderId).build();
            result.downloadBlankToken(apiTokenEncrypter.toDownloadBlankToken(token));
        }
        return result.build();
    }

    private int determineHotelGeoId(GeoHotelInfo.Coordinates coordinates) {
        var hotelGeoId = GeoBaseHelpers.getRegionIdByLocationOrNull(
                geoBase,
                coordinates.getLatitude(),
                coordinates.getLongitude()
        );
        return Objects.requireNonNullElse(hotelGeoId, 0);
    }

    private HotelOrderItem mapHotelOrder(TOrderInfo orderInfo) {
        HotelOrder orderModel = hotelOrdersService.getOrderFromProto(orderInfo);
        HotelOrderDto order = dtoMapper.buildOrderDto(orderModel, false);
        HotelOrderItem item = new HotelOrderItem();
        item.setId(order.getId().toString());

        item.setYandexOrderId(order.getYandexOrderId());
        item.setPassportId(orderInfo.getOwner().getPassportId());
        if (order.getOrderInfo().getBasicHotelInfo() != null && order.getOrderInfo().getBasicHotelInfo().getPermalink() != null) {
            item.setHotelPermalink(order.getOrderInfo().getBasicHotelInfo().getPermalink());
        }
        item.setHotelPermalink(order.getOrderInfo().getBasicHotelInfo().getPermalink());
        item.setHotelName(orderModel.getOfferInfo().getHotelInfo().getName());
        item.setAddress(order.getOrderInfo().getBasicHotelInfo().getAddress());
        item.setImageUrlTemplate(order.getOrderInfo().getBasicHotelInfo().getImageUrlTemplate());
        var coordinates = order.getOrderInfo().getBasicHotelInfo().getCoordinates();
        if (coordinates != null) {
            item.setCoordinates(coordinates);
            int hotelGeoId = determineHotelGeoId(coordinates);
            item.setGeoId(hotelGeoId);
            Integer cityGeoId = GeoBaseHelpers.getRegionRoundTo(geoBase, hotelGeoId, GeoBaseHelpers.CITY_REGION_TYPE,
                    domain);
            item.setCityGeoId(cityGeoId);
        }
        item.setConfirmationId(orderModel.getConfirmationId());
        item.setDocumentUrl(orderModel.getDocumentUrl());
        item.setNumStars(order.getOrderInfo().getBasicHotelInfo().getStars());

        item.setCheckinDate(order.getOrderInfo().getRequestInfo().getCheckinDate());
        item.setCheckoutDate(order.getOrderInfo().getRequestInfo().getCheckoutDate());
        item.setGuests(order.getGuestInfo().getGuests().stream()
                .map(g -> new Guest(g.getFirstName(), g.getLastName()))
                .collect(Collectors.toList()));
        item.setState(orderInfo.getDisplayOrderState());
        item.setStatus(DisplayableOrderStatus.fromProto(orderInfo.getDisplayOrderState()));
        item.setOrderType(OrderType.HOTEL);
        item.setFulfillmentType(hotelsFulfillmentMapping.get(order.getStatus()));
        if (order.getOrderPriceInfo() != null) {
            item.setPrice(DtoMapper.moneyToRateDto(order.getOrderPriceInfo().getPrice()));
        } else {
            item.setPrice(order.getOrderInfo().getRateInfo().getHotelCharges().getTotals().getGrand());
        }
        item.setServicedAt(ProtoUtils.toLocalDateTime(orderInfo.getServicedAt()));
        item.setSource(OrderListSource.ORCHESTRATOR);
        if (orderModel.getPayment() != null) {
            HotelOrderListItemPayment.Next next = null;
            if (orderModel.getPayment().getNext() != null &&
                    orderModel.getPayment().getNext().getPaymentEndsAt() != null &&
                    orderModel.getPayment().getNext().getPaymentEndsAt().isAfter(Instant.now())) {
                next = HotelOrderListItemPayment.Next.builder()
                        .amount(orderModel.getPayment().getNext().getAmount())
                        .paymentEndsAt(orderModel.getPayment().getNext().getPaymentEndsAt())
                        .penaltyIfUnpaid(orderModel.getPayment().getNext().getPenaltyIfUnpaid())
                        .build();
            }

            item.setPayment(HotelOrderListItemPayment.builder()
                    .amountPaid(orderModel.getPayment().getAmountPaid())
                    .usesDeferredPayments(orderModel.getPayment().isUsesDeferredPayment())
                    .next(next)
                    .mayBeStarted(orderModel.getPayment().isMayBeStarted())
                    .build());
        }
        return item;
    }

    @SuppressWarnings("Duplicates")
    private AviaOrderItem mapAviaOrder(TOrderInfo orderInfo) {
        ensureAviaExpectedOrderStructure(orderInfo);
        OrderDTO convertedOrder = aviaOrchestratorModelConverter.fromProto(orderInfo, AeroflotServicePayload.class);
        AviaOrderItem item = new AviaOrderItem();
        item.setId(convertedOrder.getId());
        item.setPassportId(orderInfo.getOwner().getPassportId());
        item.setYandexOrderId(convertedOrder.getPrettyId());
        item.setOrderType(OrderType.AVIA);
        item.setStatus(DisplayableOrderStatus.fromProto(convertedOrder.getEDisplayOrderState()));
        item.setState(orderInfo.getDisplayOrderState());
        item.setAirReservation(convertedOrder.getAirReservation());
        item.setTravellers(convertedOrder.getTravellers());
        item.setPrice(convertedOrder.getPrice());
        item.setTimeLimitAt(convertedOrder.getTimeLimitAt());
        item.setErrorCode(convertedOrder.getErrorCode());
        item.setReference(convertedOrder.getReference());
        item.setServicedAt(convertedOrder.getServicedAt());
        item.setSource(OrderListSource.ORCHESTRATOR);
        return item;
    }


    private TrainOrderItem mapTrainOrder(TOrderInfo orderInfo) {
        var item = new TrainOrderItem();
        TrainOrderListInfo trainOrderListInfo = trainModelMapService.convertToListInfo(orderInfo);
        item.setOrderInfo(trainOrderListInfo);
        item.setId(orderInfo.getOrderId());
        item.setPassportId(orderInfo.getOwner().getPassportId());
        item.setYandexOrderId(orderInfo.getPrettyId());

        // TODO (mbobrov): determine train order status
        item.setStatus(DisplayableOrderStatus.fromProto(orderInfo.getDisplayOrderState()));
        item.setState(orderInfo.getDisplayOrderState());
        item.setServicedAt(ProtoUtils.toLocalDateTime(orderInfo.getServicedAt()));
        item.setOrderType(OrderType.TRAIN);
        item.setSource(OrderListSource.ORCHESTRATOR);
        return item;
    }

    private void ensureAviaExpectedOrderStructure(TOrderInfo order) {
        Preconditions.checkArgument(order.getServiceCount() == 1,
                "exactly 1 service is expected instead of %s", order.getServiceCount());
        Preconditions.checkArgument(order.getInvoiceCount() <= 1,
                "no more than 1 invoice is expected instead of %s", order.getInvoiceCount());
    }
}
