package ru.yandex.travel.api.services.orders.trips;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import io.grpc.Context;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.models.trips.orders.OrderInfoPayloadList;
import ru.yandex.travel.api.models.trips.orders.OrderItem;
import ru.yandex.travel.api.services.orders.OrchestratorClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.TOffsetPage;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrdersInfoReq;
import ru.yandex.travel.orders.proto.TGetOrdersInfoWithoutExcludedReq;
import ru.yandex.travel.orders.proto.TOrderInfo;

import static java.util.stream.Collectors.toList;

@Service
@RequiredArgsConstructor
public class TripsOrdersService {
    private final OrchestratorClientFactory orchestratorClientFactory;
    private final OrderItemMapper travelOrderItemMapper;

    public CompletableFuture<OrderInfoPayloadList> getOrdersWithoutExcluded(
            List<String> excludedOrderIds,
            Set<EDisplayOrderState> displayOrderStates,
            Set<EDisplayOrderType> orderTypes,
            int offset,
            int limit,
            Instant requestedAt,
            UserCredentials userCredentials) {
        TGetOrdersInfoWithoutExcludedReq.Builder rqBuilder = TGetOrdersInfoWithoutExcludedReq.newBuilder()
                .addAllExcludedOrderIds(excludedOrderIds)
                .addAllDisplayOrderStates(displayOrderStates)
                .addAllTypes(orderTypes)
                .setRequestedAt(ProtoUtils.fromInstant(requestedAt))
                .setPage(TOffsetPage.newBuilder().setOffset(offset).setLimit(limit).build());

        try {
            return Context.current().withValue(UserCredentials.KEY, userCredentials).call(() -> {

                var futureOrders = FutureUtils.buildCompletableFuture(
                    orchestratorClientFactory.createRoundRobinStub().getOrdersInfoWithoutExcluded(rqBuilder.build())
                );
                return futureOrders.thenApply(orders -> {
                    var res = new OrderInfoPayloadList();
                    res.setLimit(orders.getPage().getLimit());
                    res.setOffset(orders.getPage().getOffset());
                    res.setHasMoreOrders(orders.getHasMoreOrders());
                    res.setOrders(mapOrders(orders.getResultList()));
                    return res;
                });
            });
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<List<OrderItem>> selectOrders(List<UUID> orderIds,
                                                           UserCredentials userCredentials) {
        if (orderIds.isEmpty()) {
            return CompletableFuture.completedFuture(List.of());
        }
        var request = TGetOrdersInfoReq.newBuilder()
                .addAllOrderIds(orderIds.stream().map(UUID::toString).collect(Collectors.toUnmodifiableList()))

                .build();

        try {
            return Context.current().withValue(UserCredentials.KEY, userCredentials).call(() -> {
                var futureOrders =
                        FutureUtils.buildCompletableFuture(orchestratorClientFactory.createRoundRobinStub().getOrdersInfo(request));
                return futureOrders.thenApply(orders -> mapOrders(orders.getResultList()));
            });
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<OrderItem> getOrder(UUID orderId) {
        var request = TGetOrderInfoReq.newBuilder().setOrderId(orderId.toString()).build();

        try {
            var futureOrders =
                    FutureUtils.buildCompletableFuture(orchestratorClientFactory.createOrderNoAuthFutureStub().getOrderInfo(request));
            return futureOrders.thenApply(orders -> travelOrderItemMapper.map(orders.getResult()));
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private List<OrderItem> mapOrders(List<TOrderInfo> orderInfos) {
        return orderInfos.stream()
                .map(travelOrderItemMapper::map)
                .collect(toList());
    }
}
