package ru.yandex.travel.api.services.orders.user_info;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import io.grpc.Context;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.order_type.proto.EOrderType;
import ru.yandex.travel.orders.user_info.proto.TGetUserExistingOrderTypesReq;
import ru.yandex.travel.orders.user_info.proto.TGetUserExistingOrderTypesRsp;

@Slf4j
@RequiredArgsConstructor
public class UserInfoGrpcService {
    private final UserInfoGrpcClientFactory userInfoGrpcClientFactory;
    private final UserInfoGrpcProperties properties;
    private final Retry retryHelper;
    private final Counter errorsCounter = Counter.builder("orders.userInfoGrpcService.errors").register(Metrics.globalRegistry);


    public CompletableFuture<List<EOrderType>> getUserExistingOrderTypes(UserCredentials credentials) {
        if (!properties.isEnabled()) {
            return CompletableFuture.completedFuture(Collections.emptyList());
        }

        return retryHelper.withRetry("UserInfoGrpcService::getUserExistingOrderTypes",
                () -> innerGetUserExistingOrderTypes(credentials)
                        .whenComplete((r, t) -> {
                            if (t != null) {
                                log.error("Exception on UserInfoGrpcService.getUserExistingOrderTypes, uc: {}",
                                        UserCredentials.get(), t);
                                errorsCounter.increment();
                            }
                        })
                        .thenApply(TGetUserExistingOrderTypesRsp::getExistingOrderTypesList),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    private CompletableFuture<TGetUserExistingOrderTypesRsp> innerGetUserExistingOrderTypes(UserCredentials credentials) {
        try {
            return Context.current().withValue(UserCredentials.KEY, credentials).call(() ->
                    FutureUtils.buildCompletableFuture(userInfoGrpcClientFactory.createFutureStub()
                            .getUserExistingOrderTypes(TGetUserExistingOrderTypesReq.newBuilder().build()))
            );
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }
}
