package ru.yandex.travel.api.services.portal_recipes;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.proto.portal_recipes.PortalRecipesRaw;
import ru.yandex.travel.yt_lucene_index.NativeProtobufPlainYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndexParams;

@Component
@EnableConfigurationProperties(PortalRecipesServiceProperties.class)
@Slf4j
public class PortalRecipesService {

    PortalRecipesServiceProperties params;

    public static final String AVIA_TYPE = "avia";
    public static final String BUSES_TYPE = "buses";
    public static final String HOTELS_TYPE = "hotels";
    public static final String TRAINS_TYPE = "trains";

    private static final List<NativeProtobufPlainYtLuceneIndex<PortalRecipesRaw>> indexes = new ArrayList<>();

    private static final AtomicReference<List<PortalRecipesRaw>> AviaRecipesRef = new AtomicReference<>();
    private static final AtomicReference<List<PortalRecipesRaw>> BusesRecipesRef = new AtomicReference<>();
    private static final AtomicReference<List<PortalRecipesRaw>> HotelsRecipesRef = new AtomicReference<>();
    private static final AtomicReference<List<PortalRecipesRaw>> TrainsRecipesRef = new AtomicReference<>();


    public PortalRecipesService(PortalRecipesServiceProperties parameters) {
        params = parameters;
        if (!params.isEnabled()) {
            log.warn("Portal recipes service disabled");
            return;
        }
        for (String type: List.of(AVIA_TYPE, BUSES_TYPE, HOTELS_TYPE, TRAINS_TYPE)) {
            registerIndex(type);
        }
    }

    private void registerIndex(String type) {
        var indexParams = new YtLuceneIndexParams();
        indexParams.setUpdateInterval(params.getUpdateInterval());
        indexParams.setIndexPath(params.getIndexPath() + type);
        indexParams.setTablePath(params.getTablePath() + type);
        indexParams.setShutdownTimeout(params.getShutdownTimeout());
        indexParams.setToken(params.getToken());
        indexParams.setProxy(params.getProxy());
        indexParams.setPreload(params.isPreload());
        var name = String.format("portal_recipes_%s", type);
        var index = new NativeProtobufPlainYtLuceneIndex<PortalRecipesRaw>(indexParams, name, PortalRecipesRaw.class);
        index.setIndexUpdateHandler(() -> {
            List<PortalRecipesRaw> recipes = new ArrayList<>();
            index.forEachProtoRecord(PortalRecipesRaw.parser(), (doc) -> {
                recipes.add(doc);
            });
            recipes.sort(Comparator.comparing(PortalRecipesRaw::getOrders).reversed());
            var recipesRef = getRecipesRef(type);
            recipesRef.set(recipes);
        });
        indexes.add(index);
    }

    private AtomicReference<List<PortalRecipesRaw>> getRecipesRef(String type) {
        switch (type) {
            case AVIA_TYPE:
                return AviaRecipesRef;
            case BUSES_TYPE:
                return BusesRecipesRef;
            case HOTELS_TYPE:
                return HotelsRecipesRef;
            case TRAINS_TYPE:
                return TrainsRecipesRef;
            default:
                throw new TravelApiBadRequestException("Unknown type=" + type);
        }
    }

    @PostConstruct
    public void init() {
        if (params.isEnabled()) {
            for (YtLuceneIndex index : indexes) {
                index.start();
            }
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (params.isEnabled()) {
            for (YtLuceneIndex index : indexes) {
                index.stop();
            }
        }
    }

    public boolean isReady() {
        if (!params.isEnabled()) {
            return true;
        }
        for (YtLuceneIndex index : indexes) {
            if (!index.isReady()) {
                return false;
            }
        }
        return true;
    }

    public Collection<PortalRecipesRaw> get(String type) {
        if (!params.isEnabled()) {
            throw new IllegalStateException("Portal recipes service disabled");
        }
        var recipes = getRecipesRef(type).get();
        if (recipes == null) {
            throw new IllegalStateException(String.format("Recipes for type=%s are not ready", type));
        }
        return recipes;
    }
}
