package ru.yandex.travel.api.services.promo;

import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;

import javax.money.CurrencyUnit;

import com.google.common.base.Strings;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.yandex_plus.req_rsp.GetUserYandexPlusInfoReqV1;
import ru.yandex.travel.api.endpoints.yandex_plus.req_rsp.GetUserYandexPlusInfoRspV1;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.HttpApiRetryableException;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.commons.retry.SpeculativeRetryStrategy;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustBoundPaymentMethod;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;

@Component
@RequiredArgsConstructor
@Slf4j
public class YandexPlusService {
    // actually the only supported currency at this moment
    public static final CurrencyUnit DEFAULT_PLUS_CURRENCY = ProtoCurrencyUnit.RUB;

    private final TrustClient trustClient;
    private final RetryRateLimiter rateLimiter = new RetryRateLimiter(0.3);
    private final Retry retryHelper;
    private final Counter hotelPromoTrustErrorsCounter =
            Counter.builder("hotels.promo.trust.error").register(Metrics.globalRegistry);

    public CompletableFuture<GetUserYandexPlusInfoRspV1> getUserYandexPlusInfo(
            GetUserYandexPlusInfoReqV1 req, CommonHttpHeaders headers
    ) {
        if (!headers.getUserIsPlusAsBool() || Strings.isNullOrEmpty(headers.getPassportId())) {
            return CompletableFuture.completedFuture(new GetUserYandexPlusInfoRspV1());
        }
        return getYandexPlusBalance(headers.getPassportId(), headers.getUserIP(), DEFAULT_PLUS_CURRENCY)
                .thenApply(plusPoints -> {
                    var rsp = new GetUserYandexPlusInfoRspV1();
                    rsp.setPoints(plusPoints);
                    return rsp;
                });
    }

    public CompletableFuture<Integer> getYandexPlusBalance(String passportId, String userIp, CurrencyUnit currency) {
        var retryStrategy = SpeculativeRetryStrategy.<TrustPaymentMethodsResponse>builder()
                .shouldRetryOnException(e -> e instanceof TimeoutException || e instanceof IOException || e instanceof HttpApiRetryableException)
                .timeout(Duration.ofSeconds(7))
                .retryDelay(Duration.ofMillis(500))
                .numRetries(5)
                .build();
        return retryHelper.withSpeculativeRetry("YandexPlus::getYandexPlusBalance",
                        trustClient::getPaymentMethodsAsync,
                        new TrustUserInfo(passportId, userIp),
                        retryStrategy,
                        rateLimiter)
                .thenApply(methods -> {
                    TrustBoundPaymentMethod yandexAccount = methods.getBoundYandexAccountMethod(currency);
                    return yandexAccount != null ? yandexAccount.getBalance().intValue() : null;
                });
    }

    public CompletableFuture<Integer> getYandexPlusBalanceWithoutException(String passportId, String userIp,
                                                                           CurrencyUnit currency) {
        return getYandexPlusBalance(passportId, userIp, currency)
                .exceptionally(t -> {
                    log.error("Failed to get Yandex Plus points from Trust", t);
                    hotelPromoTrustErrorsCounter.increment();
                    // we should not interrupt the booking process in any case
                    return null;
                });
    }

    public boolean isCurrencySupported(CurrencyUnit currency) {
        return DEFAULT_PLUS_CURRENCY.equals(currency);
    }
}
