package ru.yandex.travel.api.services.seo;

import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.models.common.Price;
import ru.yandex.travel.api.models.seo.FAQSchemaMarkupItem;
import ru.yandex.travel.api.models.seo.OpenGraphInfo;
import ru.yandex.travel.api.models.seo.SeoInfo;
import ru.yandex.travel.api.models.seo.blocks.ExternalLinkBlock;
import ru.yandex.travel.api.models.seo.blocks.LinkTextBlock;
import ru.yandex.travel.api.models.seo.blocks.PlainTextBlock;
import ru.yandex.travel.api.models.seo.blocks.PriceTextBlock;
import ru.yandex.travel.api.models.seo.blocks.TextBlock;
import ru.yandex.travel.api.models.seo.blocks.TextChildBlock;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.seo.proto.PlainTextStyle;

@Service
public class SeoConverter {
    public static final Map<PlainTextStyle, PlainTextBlock.Style> PLAIN_TEXT_STYLE_MAP = ImmutableMap.of(
            PlainTextStyle.BOLD, PlainTextBlock.Style.BOLD,
            PlainTextStyle.ITALIC, PlainTextBlock.Style.ITALIC
    );

    public static final Map<ProtoCurrencyUnit, Price.Currency> CURRENCY_MAP = ImmutableMap.of(
            ProtoCurrencyUnit.RUB, Price.Currency.RUB,
            ProtoCurrencyUnit.USD, Price.Currency.USD,
            ProtoCurrencyUnit.EUR, Price.Currency.EUR,
            ProtoCurrencyUnit.JPY, Price.Currency.JPY
    );

    public TextBlock convertTextBlock(ru.yandex.travel.seo.proto.TextBlock textBlock) {
        return new TextBlock(textBlock.getChildrenList().stream()
                .map(this::convertTextChildBlock)
                .collect(Collectors.toList()));
    }

    private TextChildBlock convertTextChildBlock(ru.yandex.travel.seo.proto.TextChildBlock childPb) {
        switch (childPb.getInstanceCase()) {
            case PLAIN_TEXT_BLOCK:
                return convertPlainTextBlock(childPb.getPlainTextBlock());
            case LINK_TEXT_BLOCK:
                return convertLinkTextBlock(childPb.getLinkTextBlock());
            case PRICE_TEXT_BLOCK:
                return convertPriceTextBlock(childPb.getPriceTextBlock());
            default:
                throw new RuntimeException("Not supported TextChildBlock type " + childPb.getInstanceCase().name());
        }
    }

    private PriceTextBlock convertPriceTextBlock(ru.yandex.travel.seo.proto.PriceTextBlock priceTextBlock) {
        return new PriceTextBlock(new PriceTextBlock.Data(
                convertPrice(priceTextBlock.getPrice())
        ));
    }

    private LinkTextBlock convertLinkTextBlock(ru.yandex.travel.seo.proto.LinkTextBlock linkTextBlock) {
        switch (linkTextBlock.getInstanceCase()) {
            case EXTERNAL_LINK_BLOCK:
                return convertExternalLinkTextBlock(linkTextBlock.getExternalLinkBlock());
            default:
                throw new RuntimeException("Not supported LinkTextBlock type " + linkTextBlock.getInstanceCase().name());
        }
    }

    private ExternalLinkBlock convertExternalLinkTextBlock(ru.yandex.travel.seo.proto.ExternalLinkBlock externalLinkBlock) {
        return new ExternalLinkBlock(new ExternalLinkBlock.Data(
                externalLinkBlock.getText(),
                externalLinkBlock.getUrl()
        ));
    }

    private PlainTextBlock convertPlainTextBlock(ru.yandex.travel.seo.proto.PlainTextBlock plainTextBlock) {
        return new PlainTextBlock(new PlainTextBlock.Data(
                plainTextBlock.getText(),
                plainTextBlock.getStylesList().stream()
                        .map(PLAIN_TEXT_STYLE_MAP::get)
                        .collect(Collectors.toList())
        ));
    }

    public Price convertPrice(TPrice tPrice) {
        var money = ProtoUtils.fromTPrice(tPrice);
        return new Price(money.getNumberStripped().toPlainString(),
                CURRENCY_MAP.get((ProtoCurrencyUnit) money.getCurrency()));
    }

    public FAQSchemaMarkupItem convertFAQItem(ru.yandex.travel.seo.proto.FAQSchemaMarkupItem item) {
        return new FAQSchemaMarkupItem(
                item.getQuestion(),
                item.getAnswer()
        );
    }

    public <T> SeoInfo<T> convertSeoInfo(ru.yandex.travel.seo.proto.SeoInfo seoInfoPb, T schemaOrg) {
        return new SeoInfo<T>(
                seoInfoPb.getTitle(),
                seoInfoPb.getDescription(),
                convertOpenGraphInfo(seoInfoPb.getOpenGraph()),
                schemaOrg
        );
    }

    private OpenGraphInfo convertOpenGraphInfo(ru.yandex.travel.seo.proto.OpenGraphInfo openGraph) {
        OpenGraphInfo.Size size = null;
        if (openGraph.hasImageSize()) {
            size = new OpenGraphInfo.Size(
                    openGraph.getImageSize().getWidth(),
                    openGraph.getImageSize().getHeight()
            );
        }
        return new OpenGraphInfo(
                openGraph.getTitle(),
                openGraph.getDescription(),
                openGraph.hasImage() ? openGraph.getImage() : null,
                size
        );
    }
}
