package ru.yandex.travel.api.services.seo

import org.apache.lucene.document.Document
import org.apache.lucene.document.Field
import org.apache.lucene.document.StoredField
import org.apache.lucene.document.StringField
import org.apache.lucene.index.Term
import org.apache.lucene.search.TermQuery
import ru.yandex.travel.api.proto.seo.TSeoExpListElement
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder
import ru.yandex.travel.yt_lucene_index.LuceneIndexService
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getProtoField
import java.time.ZonedDateTime

class SeoExperimentsService(
    val params: SeoExperimentsDataProviderProperties,
    luceneIndexBuilder: LuceneIndexBuilder<TSeoExpListElement>
) : LuceneIndexService<TSeoExpListElement>(params, luceneIndexBuilder, "SeoExperimentsHotelReplacements"),
    SeoExperimentsDataProvider {
    companion object {
        private const val FIELD_ID_SLUG = "s"
        private const val FIELD_PROTO_STORE = "ps"
    }

    override fun getExpReplacement(hotelSlug: String, element: SeoExperimentsElement): SeoExperimentsOption {
        if (params.experiments.isEmpty()) {
            return SeoExperimentsOption.DEFAULT
        }

        return luceneIndex.search { searcher ->
            val query = TermQuery(Term(FIELD_ID_SLUG, hotelSlug))
            val docs = searcher.search(query, 10000)

            if (docs.totalHits < 1) {
                return@search SeoExperimentsOption.DEFAULT
            }

            val now = ZonedDateTime.now()
            for (scoreDoc in docs.scoreDocs) {
                val listElement = searcher.doc(scoreDoc.doc).toSeoExpListElement()
                for (exp in params.experiments) {
                    if (exp.key == listElement.experiment && exp.value.element == element) {
                        val start: ZonedDateTime? = exp.value.start.orElse(null)
                        val stop: ZonedDateTime? = exp.value.stop.orElse(null)
                        if ((start == null || now >= start) && (stop == null || now < stop)) {
                            return@search exp.value.option
                        }
                    }
                }
            }

            return@search SeoExperimentsOption.DEFAULT
        }
    }

    override fun documentProducer(row: TSeoExpListElement): Iterable<Document> {
        val document = Document()
        document.add(StringField(FIELD_ID_SLUG, row.slug, Field.Store.NO))
        document.add(StoredField(FIELD_PROTO_STORE, row.toByteArray()))
        return listOf(document)
    }

    private fun Document.toSeoExpListElement(): TSeoExpListElement {
        return getProtoField(TSeoExpListElement.parser(), FIELD_PROTO_STORE)
    }
}
