package ru.yandex.travel.api.services.subscriptions;

import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscribeOnBetterPriceReqV1;
import ru.yandex.travel.api.models.avia.ServiceClass;
import ru.yandex.travel.api.services.subscriptions.model.better_price_subscription.Price;
import ru.yandex.travel.api.services.subscriptions.model.better_price_subscription.Tariff;
import ru.yandex.travel.notifier.subscriptions.v1.BetterPriceSubscribeReq;
import ru.yandex.travel.notifier.subscriptions.v1.Flight;
import ru.yandex.travel.notifier.subscriptions.v1.Passengers;
import ru.yandex.travel.notifier.subscriptions.v1.Variant;

@Service
public class BetterPriceSubscribeRequestMapper {
    public BetterPriceSubscribeReq map(SubscribeOnBetterPriceReqV1 request) {
        var variant = request.getVariant();
        var passengers = request.getPassengers();

        var betterPriceSubscribeReqBuilder = BetterPriceSubscribeReq.newBuilder()
                .setEmail(request.getEmail())
                .setTravelVertical(request.getTravelVerticalName().getValue())
                .setSource(request.getSource())
                .setNationalVersion(request.getNationalVersion())
                .setLanguage(request.getLanguage())
                .setUserTimezone(request.getTimezone())
                .setVariant(
                        Variant.newBuilder()
                                .addAllForwardFlights(mapFlights(variant.getForwardFlights()))
                                .addAllBackwardFlights(mapFlights(variant.getBackwardFlights())).build()
                )
                .setFromPointKey(request.getFromPointKey())
                .setToPointKey(request.getToPointKey())
                .setDateForward(request.getDateForward())
                .setPassengers(
                        Passengers.newBuilder()
                                .setAdults(passengers.getAdults())
                                .setChildren(passengers.getChildren())
                                .setInfants(passengers.getInfants())
                )
                .setServiceClass(mapServiceClass(request.getServiceClass()))
                .setWithBaggage(request.getWithBaggage());

        if (!Strings.isNullOrEmpty(request.getDateBackward())) {
            betterPriceSubscribeReqBuilder.setDateBackward(request.getDateBackward());
        }
        if (request.getTariffWithoutBaggage() != null) {
            betterPriceSubscribeReqBuilder.setTariffWithoutBaggage(mapTariff(request.getTariffWithoutBaggage()));
        }
        if (request.getTariffWithBaggage() != null) {
            betterPriceSubscribeReqBuilder.setTariffWithBaggage(mapTariff(request.getTariffWithBaggage()));
        }
        return betterPriceSubscribeReqBuilder.build();
    }

    private ru.yandex.travel.notifier.subscriptions.v1.ServiceClass mapServiceClass(ServiceClass serviceClass) {
        switch (serviceClass) {
            case ECONOMY:
                return ru.yandex.travel.notifier.subscriptions.v1.ServiceClass.SERVICE_CLASS_ECONOMY;
            case BUSINESS:
                return ru.yandex.travel.notifier.subscriptions.v1.ServiceClass.SERVICE_CLASS_BUSINESS;
            default:
                return ru.yandex.travel.notifier.subscriptions.v1.ServiceClass.UNRECOGNIZED;
        }
    }

    private List<Flight> mapFlights(List<ru.yandex.travel.api.services.subscriptions.model.better_price_subscription.Flight> flights) {
        if (flights == null) {
            return List.of();
        }
        return flights.stream()
                .map(f -> ru.yandex.travel.notifier.subscriptions.v1.Flight.newBuilder()
                        .setFromStationID(f.getFromStationId())
                        .setToStationID(f.getToStationId())
                        .setLocalDepartureDatetime(f.getLocalDepartureDatetime())
                        .setCompanyCode(f.getCompanyCode())
                        .setNumber(f.getNumber())
                        .build()
                )
                .collect(Collectors.toUnmodifiableList());
    }

    private ru.yandex.travel.notifier.subscriptions.v1.Tariff mapTariff(Tariff tariff) {
        var tariffBuilder = ru.yandex.travel.notifier.subscriptions.v1.Tariff.newBuilder();
        if (tariff.getPrice() != null) {
            tariffBuilder.setPrice(mapPrice(tariff.getPrice()));
        }
        if (tariff.getExchangedPrice() != null) {
            tariffBuilder.setPrice(mapPrice(tariff.getExchangedPrice()));
        }
        return tariffBuilder.build();
    }

    private ru.yandex.travel.notifier.subscriptions.v1.Price mapPrice(Price price) {
        return ru.yandex.travel.notifier.subscriptions.v1.Price.newBuilder()
                .setCurrency(price.getCurrency())
                .setValue(price.getValue())
                .build();
    }
}
