package ru.yandex.travel.api.services.subscriptions;

import java.util.concurrent.CompletableFuture;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.CredentialsTuple;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscribeOnBetterPriceReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscribeReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionPromoConfigReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionPromoConfigRspV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionStatusReqV1;
import ru.yandex.travel.api.endpoints.subscriptions.req_rsp.SubscriptionStatusRspV1;
import ru.yandex.travel.api.services.notifier.NotifierClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.notifier.subscriptions.v1.GetPromoConfigReq;
import ru.yandex.travel.notifier.subscriptions.v1.GetStatusReq;
import ru.yandex.travel.notifier.subscriptions.v1.SubscribeReq;
import ru.yandex.travel.notifier.subscriptions.v1.UnsubscribeReq;

@Service
@RequiredArgsConstructor
@Slf4j
public class NotifierSubscriptionServiceImpl implements NotifierSubscriptionService {
    private final NotifierClientFactory notifierClientFactory;
    private final BetterPriceSubscribeRequestMapper betterPriceSubscribeRequestMapper;

    @Override
    public CompletableFuture<Void> subscribe(SubscribeReqV1 request, CommonHttpHeaders commonHttpHeaders) {
        // Workaround for undocumented fields in request. To be removed after frontend fix
        fillCredentials(request);

        var subscribeReqBuilder = SubscribeReq.newBuilder()
                .setEmail(request.getEmail())
                .setVertical(request.getTravelVerticalName().getValue())
                .setSource(request.getSource())
                .setNationalVersion(request.getNationalVersion())
                .setLanguage(request.getLanguage())
                .setTimezone(request.getTimezone())
                .setPassportId(Strings.nullToEmpty(request.getPassportId()))
                .setYandexUid(Strings.nullToEmpty(request.getYandexUid()));

        if (commonHttpHeaders != null) {
            subscribeReqBuilder
                    .setIsPlusUser(commonHttpHeaders.getUserIsPlusAsBool())
                    .putAllExperiments(commonHttpHeaders.getExperiments());
        }

        var notifierClient = notifierClientFactory.createRoundRobinStub();
        return FutureUtils.buildCompletableFuture(notifierClient.subscribe(subscribeReqBuilder.build())).thenApply(rsp -> null);
    }

    @Override
    public CompletableFuture<Void> unsubscribe(String hash) {
        var unsubscribeReq = UnsubscribeReq.newBuilder().setHash(hash).build();
        var notifierClient = notifierClientFactory.createRoundRobinStub();
        return FutureUtils.buildCompletableFuture(notifierClient.unsubscribe(unsubscribeReq)).thenApply(rsp -> null);
    }

    @Override
    public CompletableFuture<SubscriptionStatusRspV1> getStatus(SubscriptionStatusReqV1 request) {
        var getStatusReq = GetStatusReq.newBuilder().setEmail(request.getEmail()).build();
        var notifierClient = notifierClientFactory.createRoundRobinStub();
        return FutureUtils.buildCompletableFuture(notifierClient.getStatus(getStatusReq))
                .thenApply(rsp ->
                        SubscriptionStatusRspV1.builder()
                                .isSubscribed(rsp.getIsSubscribed())
                                .source(rsp.getSource())
                                .vertical(rsp.getVertical())
                                .subscribedAt(rsp.getSubscribedAt())
                                .unsubscribedAt(rsp.getUnsubscribedAt())
                                .build());
    }

    @Override
    public CompletableFuture<SubscriptionPromoConfigRspV1> getPromoConfig(SubscriptionPromoConfigReqV1 request,
                                                                          CommonHttpHeaders commonHttpHeaders) {
        var getPromoConfigRequest = GetPromoConfigReq.newBuilder()
                .setVertical(request.getVertical().name())
                .setIsPlusUser(commonHttpHeaders.getUserIsPlusAsBool())
                .build();
        var notifierClient = notifierClientFactory.createRoundRobinStub();
        return FutureUtils.buildCompletableFuture(notifierClient.getPromoConfig(getPromoConfigRequest))
                .thenApply(SubscriptionPromoConfigRspV1::fromProto);
    }

    @Override
    public CompletableFuture<Void> subscribeOnBetterPrice(SubscribeOnBetterPriceReqV1 request) {
        var subscribeOnBetterPriceRequest = betterPriceSubscribeRequestMapper.map(request);
        var notifierClient = notifierClientFactory.createRoundRobinStub();
        return FutureUtils
                .buildCompletableFuture(notifierClient.subscribeOnBetterPrice(subscribeOnBetterPriceRequest))
                .thenApply(rsp -> null);
    }

    private void fillCredentials(SubscribeReqV1 request) {
        if (request.getCredentials() == null) {
            return;
        }
        for (CredentialsTuple credentialsItem : request.getCredentials()) {
            switch (credentialsItem.key) {
                case "yandexUid": request.setYandexUid(credentialsItem.value);
                case "passportId": request.setPassportId(credentialsItem.value);
            }
        }
    }
}
