package ru.yandex.travel.api.services.train;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.Param;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.jackson.MoneySerializersModule;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.tvm.TvmWrapper;

@Service
@EnableConfigurationProperties(TrainApiProperties.class)
@Slf4j
public class TrainApiProxy {
    private static final Set<Integer> SUCCESSFUL_RESPONSE_CODES = ImmutableSet.of(200, 201);

    private final AsyncHttpClientWrapper asyncHttpClient;
    private final TrainApiProperties config;
    private final TvmWrapper tvm;

    private ObjectMapper objectMapper;

    // in fact tvm should be enabled in all environments, made optional only for tests
    public TrainApiProxy(
            @Qualifier(value = "trainsProxyAhcWrapper") AsyncHttpClientWrapper asyncHttpClient,
            TrainApiProperties config, @Autowired(required = false) TvmWrapper tvm) {
        this.asyncHttpClient = asyncHttpClient;
        this.config = config;
        this.tvm = tvm;
        objectMapper = new ObjectMapper()
                .registerModule(new MoneySerializersModule())
                .registerModule(new JavaTimeModule());
        if (tvm != null) {
            tvm.validateAlias(config.getTvmAlias());
        }
    }

    public CompletableFuture<TrainProxyOrderListDTO> listTrainOrders(int offset, int limit,
                                                                     TrainProxyOrderStatus orderStatus,
                                                                     String searchTerm) {
        String callUrl =
                "/ru/api/user-orders/?format=json&offset=" + offset + "&limit=" + limit + "&findString" + searchTerm;
        List<Param> queryParams = new ArrayList<>();
        queryParams.add(new Param("format", "json"));
        queryParams.add(new Param("offset", Integer.toString(offset)));
        queryParams.add(new Param("limit", Integer.toString(limit)));
        queryParams.add(new Param("findString", searchTerm));
        if (orderStatus != null) {
            queryParams.add(new Param("travelStatus", orderStatus.getValue()));
        }
        return getRequest(
                callUrl, queryParams, TrainProxyOrderListDTO.class
        );
    }

    public CompletableFuture<TrainProxyAuthDataResponseDTO> getTrainOrder(String orderId, String prettyId) {
        String callUrl = "/ru/api/travel/order-auth-details/";
        List<Param> queryParams = new ArrayList<>(1);
        if (orderId != null) {
            queryParams.add(new Param("order_uid", orderId));
        } else {
            queryParams.add(new Param("express_order_number", prettyId));
        }
        return getRequest(callUrl, queryParams, TrainProxyAuthDataResponseDTO.class);
    }

    private <RS> CompletableFuture<RS> getRequest(String path, List<Param> queryParams, Class<RS> responseType) {
        RequestBuilder requestBuilder = createBaseRequestBuilder("GET")
                .setUrl(config.getBaseUrl() + path)
                .addQueryParams(queryParams);
        return asyncHttpClient.executeRequest(requestBuilder)
                .toCompletableFuture()
                .thenApply(r -> parseResponse(r, responseType));
    }

    private <T> T parseResponse(Response response, Class<T> resultClass) {
        int sc = response.getStatusCode();
        String content = response.getResponseBody();
        if (SUCCESSFUL_RESPONSE_CODES.contains(sc)) {
            try {
                return objectMapper.readValue(content, resultClass);
            } catch (IOException e) {
                log.error("Unable to parse response", e);
                throw new RuntimeException("Unable to parse response", e);
            }
        } else {
            throw new TrainApiException("Got response from train-api with status code: " + sc, sc, readJson(content));
        }
    }

    private Map readJson(String json) {
        try {
            return objectMapper.readValue(json, Map.class);
        } catch (IOException e) {
            return ImmutableMap.of("non_parsable_content", json);
        }
    }

    private RequestBuilder createBaseRequestBuilder(String method) {
        UserCredentials userCredentials = UserCredentials.get();
        RequestBuilder builder = new RequestBuilder()
                .setReadTimeout(Math.toIntExact(config.getHttpReadTimeout().toMillis()))
                .setRequestTimeout(Math.toIntExact(config.getHttpRequestTimeout().toMillis()))
                .setMethod(method);
        if (tvm != null) {
            String serviceTicket = tvm.getServiceTicket(config.getTvmAlias());
            builder.setHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), serviceTicket);
        }
        if (userCredentials != null && userCredentials.isLoggedIn()) {
            if (!Strings.isNullOrEmpty(userCredentials.getUserTicket())) {
                builder.setHeader(CommonHttpHeaders.HeaderType.USER_TICKET.getHeader(), userCredentials.getUserTicket());
            }
        }
        return builder;
    }
}
