package ru.yandex.travel.api.services.train;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.asynchttpclient.util.HttpConstants;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.models.train.CrossLink;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.AhcHttpRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;

@Service
@Slf4j
@EnableConfigurationProperties(TrainCrosslinkProperties.class)
public class TrainCrosslinkService {
    private final AsyncHttpClientWrapper ahcClient;
    private final TrainCrosslinkProperties config;
    private final Retry retryHelper;

    public TrainCrosslinkService(@Qualifier("trainCrosslinkAhcWrapper") AsyncHttpClientWrapper ahcClient,
                                 TrainCrosslinkProperties config,
                                 Retry retryHelper) {
        this.ahcClient = ahcClient;
        this.config = config;
        this.retryHelper = retryHelper;
    }

    public CompletableFuture<CrossLink[]> crosslinks(String fromKey, String toKey, String transportType, String requestId) {
        String url = config.getBaseUrl() + config.getCrosslinksPath();

        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod(HttpConstants.Methods.GET)
                .setUrl(url)
                .addQueryParam("from_key", fromKey)
                .addQueryParam("to_key", toKey)
                .addQueryParam("transport_type", transportType);

        return retryHelper.withRetry(
                "TrainCrosslinkService::crosslinks",
                req -> ahcClient.executeRequest(req, "trainCrosslinks", requestId),
                requestBuilder,
                new AhcHttpRetryStrategy(Duration.ofSeconds(3), 2)
        ).thenApply(this::parseSearchResponse);
    }

    private CrossLink[] parseSearchResponse(Response response) {
        if (response.getStatusCode() != HttpStatus.OK.value()) {
            log.error("TrainCrosslinkService::parseSearchResponse returned non-200 status code: {}", response.getStatusCode());
            throw new RuntimeException("Bad backend response code");
        }
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.USE_JAVA_ARRAY_FOR_JSON_ARRAY, true);
        try {
            return objectMapper.readValue(response.getResponseBody(), CrossLink[].class);
        } catch (JsonProcessingException e) {
            log.error("TrainCrosslinkService::parseSearchResponse bad response: ", e);
            throw new RuntimeException("Bad backend response body");
        }
    }
}
