package ru.yandex.travel.api.services.train;

import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.asynchttpclient.util.HttpConstants;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.models.rasp.morda_backend.ParseContext;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.AhcHttpRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;

@Service
@Slf4j
@EnableConfigurationProperties(TrainHttpProxyCacheProperties.class)
public class TrainHttpProxyCacheService {
    private final AsyncHttpClientWrapper ahcClient;
    private final TrainHttpProxyCacheProperties config;
    private final Retry retryHelper;

    public TrainHttpProxyCacheService(@Qualifier("trainHttpProxyCacheAhcWrapper") AsyncHttpClientWrapper ahcClient,
                                      TrainHttpProxyCacheProperties config,
                                      Retry retryHelper) {
        this.ahcClient = ahcClient;
        this.config = config;
        this.retryHelper = retryHelper;
    }

    public CompletableFuture<ParseContext> parseContext(String fromSlag, String toSlag, String nationalVersion, String transportType, String requestId) {
        String url = config.getBaseUrl() +  config.getMordaBackendParseContextPath();

        RequestBuilder requestBuilder = new RequestBuilder()
                .setMethod(HttpConstants.Methods.GET)
                .setUrl(url)
                .addQueryParam("from_slug", fromSlag)
                .addQueryParam("to_slug", toSlag)
                .addQueryParam("national_version", nationalVersion)
                .addQueryParam("t_type", transportType);

        return retryHelper.withRetry(
                "CachedMordaBackndService::parseContext",
                req -> ahcClient.executeRequest(req, "trainParseContext", requestId),
                requestBuilder,
                new AhcHttpRetryStrategy()
            ).thenApply(this::parseSearchResponse);
    }

    private ParseContext parseSearchResponse(Response response) {
        if (response.getStatusCode() != HttpStatus.OK.value()) {
            log.error("CachedMordaBackndService::parseSearchResponse returned non-200 status code: {}", response.getStatusCode());
            throw new RuntimeException("Bad backend response code");
        }
        ObjectMapper objectMapper = new ObjectMapper();
        try {
            return objectMapper.readValue(response.getResponseBody(), ParseContext.class);
        } catch (JsonProcessingException e) {
            log.error("CachedMordaBackndService::parseSearchResponse bad response: ", e);
            throw new RuntimeException("Bad backend response body");
        }
    }
}
