package ru.yandex.travel.api.services.train;

import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.trains_search.req_rsp.DirectionRsp;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlock;
import ru.yandex.travel.api.models.seo.AggregateOfferData;
import ru.yandex.travel.api.models.seo.SeoInfo;
import ru.yandex.travel.api.models.train.seo.TrainDirectionFAQBlock;
import ru.yandex.travel.api.models.train.seo.TrainDirectionFAQItem;
import ru.yandex.travel.api.models.train.seo.TrainDirectionHeaderBlock;
import ru.yandex.travel.api.models.train.seo.TrainDirectionInfoBlock;
import ru.yandex.travel.api.models.train.seo.TrainDirectionSegmentsBlock;
import ru.yandex.travel.api.models.train.seo.TrainDirectionSeoInfo;
import ru.yandex.travel.api.services.seo.SeoConverter;
import ru.yandex.travel.trains.search_api.api.seo_direction.SeoDirectionResponse;

@Service
@RequiredArgsConstructor
public class TrainSeoConverter {
    private final SeoConverter seoConverter;

    public DirectionRsp convertSeoDirection(SeoDirectionResponse protoRsp, CrosslinksHotelsBlock hotelsBlock) {
        return new DirectionRsp(
                Strings.isNullOrEmpty(protoRsp.getSearchDate()) ? null : protoRsp.getSearchDate(),
                new DirectionRsp.Blocks(
                        convertHeaderBlock(protoRsp.getHeader()),
                        protoRsp.hasSegments() ? convertSegmentsBlock(protoRsp.getSegments()) : null,
                        protoRsp.hasInfo() ? convertDirectionInfoBlock(protoRsp.getInfo()) : null,
                        protoRsp.hasFaq() ? convertDirectionFAQBlock(protoRsp.getFaq()) : null,
                        hotelsBlock
                ),
                convertSeoInfo(protoRsp.getSeoInfo(), protoRsp.getTrainSeoInfo())
        );
    }

    private SeoInfo<TrainDirectionSeoInfo> convertSeoInfo(ru.yandex.travel.seo.proto.SeoInfo seoInfoPb,
                                                          ru.yandex.travel.trains.search_api.api.seo_direction.TrainDirectionSeoInfo trainSeoInfoPb) {
        AggregateOfferData offer = null;
        if (trainSeoInfoPb.hasOfferData()) {
            offer = new AggregateOfferData(
                    trainSeoInfoPb.getOfferData().getName(),
                    seoConverter.convertPrice(trainSeoInfoPb.getOfferData().getPrice())
            );
        }
        var trainSeoInfo = new TrainDirectionSeoInfo(trainSeoInfoPb.getFaqItemsList().stream()
                .map(seoConverter::convertFAQItem).collect(Collectors.toList()),
                offer);
        return seoConverter.convertSeoInfo(seoInfoPb, trainSeoInfo);
    }

    private TrainDirectionFAQBlock convertDirectionFAQBlock(ru.yandex.travel.trains.search_api.api.seo_direction.TrainDirectionFAQBlock faq) {
        return new TrainDirectionFAQBlock(faq.getItemsList().stream()
                .map(i -> new TrainDirectionFAQItem(i.getTitle(), seoConverter.convertTextBlock(i.getText())))
                .collect(Collectors.toList()));
    }

    private TrainDirectionInfoBlock convertDirectionInfoBlock(ru.yandex.travel.trains.search_api.api.seo_direction.TrainDirectionInfoBlock info) {
        return new TrainDirectionInfoBlock(info.getTitle(),
                seoConverter.convertTextBlock(info.getText()),
                seoConverter.convertTextBlock(info.getDisclaimer()));
    }

    private TrainDirectionSegmentsBlock convertSegmentsBlock(ru.yandex.travel.trains.search_api.api.seo_direction.TrainDirectionSegmentsBlock segments) {
        return new TrainDirectionSegmentsBlock(segments.getTitle());
    }

    private TrainDirectionHeaderBlock convertHeaderBlock(ru.yandex.travel.trains.search_api.api.seo_direction.TrainDirectionHeaderBlock protoHeader) {
        return new TrainDirectionHeaderBlock(protoHeader.getTitle(),
                seoConverter.convertTextBlock(protoHeader.getText()));
    }
}
