package ru.yandex.travel.api.services.train.canonical;

import java.util.Collections;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(TrainCanonicalIndexProperties.class)
@Slf4j
public class TrainCanonicalIndex {
    private static final int INDEX_STRUCTURE_VERSION = 1;

    private static final String FIELD_FROM_SLUG = "from_slug";
    private static final String FIELD_TO_SLUG = "to_slug";

    private static final String FIELD_CANONICAL_FROM_SLUG = "canonical_from_slug";
    private static final String FIELD_CANONICAL_TO_SLUG = "canonical_to_slug";
    private static final String FIELD_CANONICAL_FROM_TITLE = "canonical_from_title";
    private static final String FIELD_CANONICAL_TO_TITLE = "canonical_to_title";
    private static final String FIELD_CANONICAL_FROM_POPULAR_TITLE = "canonical_from_popular_title";
    private static final String FIELD_CANONICAL_TO_POPULAR_TITLE = "canonical_to_popular_title";

    private final YtLuceneIndex index;

    public TrainCanonicalIndex(TrainCanonicalIndexProperties params) {
        if (params.isEnabled()) {
            index = new YsonYtLuceneIndex(params, "TrainSearchCanonicalIndex", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                final String fromSlug = row.getString(FIELD_FROM_SLUG);
                final String toSlug = row.getString(FIELD_TO_SLUG);
                document.add(new StringField(FIELD_FROM_SLUG, fromSlug, Field.Store.YES));
                document.add(new StringField(FIELD_TO_SLUG, toSlug, Field.Store.YES));

                final String fromCanonicalSlug = row.getString(FIELD_CANONICAL_FROM_SLUG);
                final String toCanonicalSlug = row.getString(FIELD_CANONICAL_TO_SLUG);
                document.add(new StoredField(FIELD_CANONICAL_FROM_SLUG, fromCanonicalSlug));
                document.add(new StoredField(FIELD_CANONICAL_TO_SLUG, toCanonicalSlug));

                final String fromTitle = row.getString(FIELD_CANONICAL_FROM_TITLE);
                final String toTitle = row.getString(FIELD_CANONICAL_TO_TITLE);
                document.add(new StoredField(FIELD_CANONICAL_FROM_TITLE, fromTitle));
                document.add(new StoredField(FIELD_CANONICAL_TO_TITLE, toTitle));

                final String fromPopularTitle = row.getString(FIELD_CANONICAL_FROM_POPULAR_TITLE);
                final String toPopularTitle = row.getString(FIELD_CANONICAL_TO_POPULAR_TITLE);
                document.add(new StoredField(FIELD_CANONICAL_FROM_POPULAR_TITLE, fromPopularTitle));
                document.add(new StoredField(FIELD_CANONICAL_TO_POPULAR_TITLE, toPopularTitle));


                return Collections.singletonList(document);
            });
        } else {
            log.warn("Train search canonical index disabled");
            index = null;
        }
    }

    @PostConstruct
    public void init() {
        if (index != null) {
            index.start();
        }
    }

    @PreDestroy
    public void destroy() {
        if (index != null) {
            index.stop();
        }
    }

    private boolean isReady() {
        return index != null && index.isReady();
    }

    public TrainCanonicalItem getCanonicalBySlug(String fromSlug, String toSlug) {
        if (!isReady()) {
            return null;
        }

        return index.search(searcher -> {
            Query query_from = new TermQuery(new Term(FIELD_FROM_SLUG, fromSlug));
            Query query_to = new TermQuery(new Term(FIELD_TO_SLUG, toSlug));
            Query query = new BooleanQuery.Builder()
                    .add(new BooleanClause(query_from, BooleanClause.Occur.MUST))
                    .add(new BooleanClause(query_to, BooleanClause.Occur.MUST))
                    .build();

            TopDocs topDocs = searcher.search(query, 1);

            if (topDocs.totalHits > 0) {
                Document document = searcher.doc(topDocs.scoreDocs[0].doc);

                return TrainCanonicalItem.builder()
                        .fromSlug(document.getField(FIELD_CANONICAL_FROM_SLUG).stringValue())
                        .toSlug(document.getField(FIELD_CANONICAL_TO_SLUG).stringValue())
                        .fromTitle(document.getField(FIELD_CANONICAL_FROM_TITLE).stringValue())
                        .toTitle(document.getField(FIELD_CANONICAL_TO_TITLE).stringValue())
                        .fromPopularTitle(document.getField(FIELD_CANONICAL_FROM_POPULAR_TITLE).stringValue())
                        .toPopularTitle(document.getField(FIELD_CANONICAL_TO_POPULAR_TITLE).stringValue())
                        .build();
            }

            return null;
        });
    }
}
