package ru.yandex.travel.api.services.train.crosslinks;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.DoubleDocValuesField;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.SortedNumericSortField;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TotalHitCountCollector;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(TrainCrossLinksIndexProperties.class)
@Slf4j
public class TrainCrossLinksIndex {
    private static final int INDEX_STRUCTURE_VERSION = 1;

    private static final String FIELD_FROM_ID = "from_key";
    private static final String FIELD_TO_ID = "to_key";

    private static final String FIELD_LINK_FROM_ID = "crosslink_from_key";
    private static final String FIELD_LINK_TO_ID = "crosslink_to_key";
    private static final String FIELD_LINK_FROM_SLUG = "crosslink_from_slug";
    private static final String FIELD_LINK_TO_SLUG = "crosslink_to_slug";
    private static final String FIELD_LINK_FROM_NOMINATIVE = "crosslink_from_nominative";
    private static final String FIELD_LINK_TO_NOMINATIVE = "crosslink_to_nominative";

    private static final String FIELD_RELEVANCE = "crosslink_relevance";

    private final YtLuceneIndex index;


    public TrainCrossLinksIndex(TrainCrossLinksIndexProperties params) {
        if (params.isEnabled()) {
            index = new YsonYtLuceneIndex(params, "TrainSearchLinksIndex", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                final String fromId = row.getString(FIELD_FROM_ID);
                final String toId = row.getString(FIELD_TO_ID);
                document.add(new StringField(FIELD_FROM_ID, fromId, Field.Store.YES));
                document.add(new StringField(FIELD_TO_ID, toId, Field.Store.YES));

                final String fromLinkId = row.getString(FIELD_LINK_FROM_ID);
                final String toLinkId = row.getString(FIELD_LINK_TO_ID);
                document.add(new StoredField(FIELD_LINK_FROM_ID, fromLinkId));
                document.add(new StoredField(FIELD_LINK_TO_ID, toLinkId));

                final String fromLinkNominative = row.getString(FIELD_LINK_FROM_NOMINATIVE);
                final String toLinkNominative = row.getString(FIELD_LINK_TO_NOMINATIVE);
                document.add(new StoredField(FIELD_LINK_FROM_NOMINATIVE, fromLinkNominative));
                document.add(new StoredField(FIELD_LINK_TO_NOMINATIVE, toLinkNominative));

                final String fromLinkSlug = row.getString(FIELD_LINK_FROM_SLUG);
                final String toLinkSlug = row.getString(FIELD_LINK_TO_SLUG);
                document.add(new StoredField(FIELD_LINK_FROM_SLUG, fromLinkSlug));
                document.add(new StoredField(FIELD_LINK_TO_SLUG, toLinkSlug));

                final double relevance = row.getDouble(FIELD_RELEVANCE);
                document.add(new StoredField(FIELD_RELEVANCE, relevance));
                document.add(new DoubleDocValuesField(FIELD_RELEVANCE, relevance));

                return Collections.singletonList(document);
            });
        } else {
            log.warn("Train search links index disabled");
            index = null;
        }
    }

    @PostConstruct
    public void init() {
        if (index != null) {
            index.start();
        }
    }

    @PreDestroy
    public void destroy() {
        if (index != null) {
            index.stop();
        }
    }

    private boolean isReady() {
        return index != null && index.isReady();
    }

    public List<TrainCrossLinkItem> getCrossLinksByFromId(String fromKey, String toKey) {
        if (!isReady()) {
            return null;
        }

        List<TrainCrossLinkItem> results = new ArrayList<>();
        return index.search(searcher -> {
            Query query_from = new TermQuery(new Term(FIELD_FROM_ID, fromKey));
            Query query_to = new TermQuery(new Term(FIELD_TO_ID, toKey));
            Query query = new BooleanQuery.Builder()
                    .add(new BooleanClause(query_from, BooleanClause.Occur.MUST))
                    .add(new BooleanClause(query_to, BooleanClause.Occur.MUST))
                    .build();

            TotalHitCountCollector collector = new TotalHitCountCollector();
            searcher.search(query, collector);
            int totalHits =  collector.getTotalHits();

            if (totalHits > 0) {
                SortField relevanceSort = new SortedNumericSortField(FIELD_RELEVANCE, SortField.Type.DOUBLE, true);
                Sort sort = new Sort(relevanceSort);

                TopDocs topDocs = searcher.search(query, totalHits, sort);

                for (ScoreDoc scoreDoc : topDocs.scoreDocs) {
                    Document document = searcher.doc(scoreDoc.doc);
                    results.add(TrainCrossLinkItem.builder()
                            .fromTitleNominative(document.getField(FIELD_LINK_FROM_NOMINATIVE).stringValue())
                            .toTitleNominative(document.getField(FIELD_LINK_TO_NOMINATIVE).stringValue())
                            .fromSlug(document.getField(FIELD_LINK_FROM_SLUG).stringValue())
                            .toSlug(document.getField(FIELD_LINK_TO_SLUG).stringValue())
                            .fromKey(document.getField(FIELD_LINK_FROM_ID).stringValue())
                            .toKey(document.getField(FIELD_LINK_TO_ID).stringValue())
                            .relevance(document.getField(FIELD_RELEVANCE).numericValue().doubleValue())
                            .build()
                    );
                }

                return results;
            }
            return null;
        });
    }
}
