package ru.yandex.travel.api.services.trips;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import io.grpc.Context;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.komod.trips.api.trips.v1.GetTripIdByOrderIdReq;


@Slf4j
@Component
@RequiredArgsConstructor
public class TripsProvider {
    private final TripsClientFactory tripsClientFactory;

    public CompletableFuture<UUID> getTripIdByOrderId(UUID orderId, UserCredentials userCredentials) {
        try {
            return Context.current().withValue(UserCredentials.KEY, userCredentials).call(() -> {
                var client = tripsClientFactory.createRoundRobinStub();
                var tripFuture = client.getTripIdByOrderId(
                        GetTripIdByOrderIdReq.newBuilder()
                                .setOrderId(orderId.toString())
                                .build());
                return FutureUtils.buildCompletableFuture(tripFuture)
                        .thenApply(tripRsp -> {
                            if (tripRsp == null || tripRsp.getTripId().isEmpty()) {
                                return null;
                            }
                            return UUID.fromString(tripRsp.getTripId());
                        }).exceptionally(e -> {
                            log.error("Unable to get tripId by orderId={}", orderId, e);
                            return null;
                        });
            });
        } catch (Exception e) {
            log.error("Unable to get tripId by orderId={}", orderId, e);
            return CompletableFuture.completedFuture(null);
        }
    }
}
