package ru.yandex.travel.api.services.weather;

import java.time.LocalDate;
import java.time.Month;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.collect.ImmutableSet;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.weather.model.GraphQLRequestData;
import ru.yandex.travel.api.services.weather.model.GraphQLRequestVariables;
import ru.yandex.travel.api.services.weather.model.GraphQLResponse;
import ru.yandex.travel.api.services.weather.model.WeatherByGeoIdData;
import ru.yandex.travel.api.services.weather.model.WeatherByGeoIdGraphQLData;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.HttpApiClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.tvm.TvmWrapper;

@Service
@Slf4j
@EnableConfigurationProperties(WeatherServiceProperties.class)
public class WeatherService extends HttpApiClientBase {
    public static final String WEATHER_BY_GEO_ID_QUERY = "query ForAvia(" +
            "  $geoId: Int!" +
            "  $climateOffset: Int!" +
            ") {" +
            "  weatherByGeoId(request: {geoId: $geoId}) {" +
            "    url" +
            "    forecast {" +
            "      days(offset: 0, limit: 11) {" +
            "        " +
            "        summary {" +
            "          day {" +
            "            humidity" +
            "            waterTemperature            " +
            "            temperature" +
            "            minTemperature" +
            "            icon(format: PNG_128)" +
            "          }          " +
            "        }" +
            "      }" +
            "    }" +
            "    " +
            "    climate {" +
            "      months(offset: $climateOffset, limit: 12, loop: true) {" +
            "        minNightTemperature" +
            "        maxDayTemperature" +
            "        precDays" +
            "        overcastDays" +
            "        humidity" +
            "        waterTemperature        " +
            "      }" +
            "    }" +
            "  }" +
            "}";

    private final GeoBase geoBase;
    private final TvmWrapper tvm;
    private final String tvmAlias;
    private final boolean tvmEnabled;
    private static final int forecastCountDays = 10;
    private static final int maxBlocksCount = 5;
    private static final Set<Integer> LOCATIONS_TYPES = ImmutableSet
            .of(GeoBaseHelpers.CITY_REGION_TYPE, GeoBaseHelpers.VILLAGE_REGION_TYPE);

    @EqualsAndHashCode(callSuper = true)
    @ToString(callSuper = true)
    @Data
    public static class WeatherByGeoIdRequest extends GraphQLRequestVariables {
        private Integer geoId;
        private Integer climateOffset;
    }

    public WeatherService(@Qualifier("weatherAsyncHttpClientWrapper") AsyncHttpClientWrapper client,
                          GeoBase geoBase,
                          WeatherServiceProperties config,
                          @Autowired(required = false) TvmWrapper tvm
                         ) {
        super(client, config, new ObjectMapper()
                .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .registerModule(new JavaTimeModule()));
        this.geoBase = geoBase;
        this.tvm = tvm;
        this.tvmAlias = config.getTvmAlias();
        this.tvmEnabled = config.getTvmEnabled();
        if (tvmEnabled && tvm != null) {
            tvm.validateAlias(tvmAlias);
        }
    }

    public CompletableFuture<Optional<WeatherByGeoIdData>> getWeatherByGeoId(Integer geoId,
                                                                   LocalDate startDate,
                                                                   LocalDate endDate) {

        var regionType = GeoBaseHelpers.getRegionTypeId(geoBase, "", geoId);
        if (!LOCATIONS_TYPES.contains(regionType)) {
            log.warn("Weather::getWeatherByGeoId: Bad region with id=" + geoId + " and type=" + regionType);
            return CompletableFuture.completedFuture(Optional.empty());
        }

        var variables = new WeatherByGeoIdRequest();
        variables.setGeoId(geoId);
        variables.setClimateOffset(LocalDate.now().getMonth().getValue() - 1);

        return executeGraphQLRequest(WEATHER_BY_GEO_ID_QUERY, variables, GraphQLResponse.class)
                .thenApply(response -> {
                    if (response.getData() != null) {
                        return Optional.of(filterWeatherByInterval(
                                mapWeatherByGeoIdData(response.getData()),
                                startDate,
                                endDate));
                    }

                    if (!CollectionUtils.isEmpty(response.getErrors())) {
                        throw new RuntimeException("Weather::getWeatherByGeoId returned errors: " + response.getErrors().toString());
                    }
                    throw new RuntimeException("Weather::getWeatherByGeoId empty data");
                });
    }

    private WeatherByGeoIdData filterWeatherByInterval(WeatherByGeoIdData data, LocalDate startDate, LocalDate endDate) {
        LinkedList<WeatherByGeoIdData.ForecastItem> forecastItems = new LinkedList<>();
        for (WeatherByGeoIdData.ForecastItem item : data.forecastItems) {
            var date = item.getDate();
            if (date.isBefore(startDate) || endDate.isBefore(date)) {
                continue;
            }
            forecastItems.add(item);
        }
        Set<Month> climateMonths = getClimateMonths(startDate, endDate);

        while (forecastItems.size() > 0) {
            if (forecastItems.size() + climateMonths.size() <= maxBlocksCount) {
                break;
            }
            var lastItem = forecastItems.getLast();
            forecastItems.removeLast();
            climateMonths.add(lastItem.getDate().getMonth());
        }
        var climateItems = data.climateItems;
        if (climateItems.size() > maxBlocksCount) {
            climateItems = climateItems.subList(0, maxBlocksCount);
        }
        return WeatherByGeoIdData.builder()
                .url(data.url)
                .forecastItems(forecastItems.stream().collect(Collectors.toUnmodifiableList()))
                .climateItems(climateItems.subList(0, maxBlocksCount).stream()
                        .filter(climateItem -> climateMonths.contains(climateItem.getMonth()))
                        .collect(Collectors.toUnmodifiableList()))
                .build();
    }

    private Set<Month> getClimateMonths(LocalDate startDate, LocalDate endDate) {
        var afterForecastDate = LocalDate.now().plusDays(forecastCountDays + 1);
        if (afterForecastDate.isAfter(startDate)) {
            startDate = afterForecastDate;
        }
        Set<Month> months = new HashSet<>();
        if (startDate.isAfter(endDate)) {
            return months;
        }

        startDate = startDate.withDayOfMonth(1);
        while (startDate.isBefore(endDate) || startDate.isEqual(endDate)) {
            months.add(startDate.getMonth());
            startDate = startDate.plusMonths(1);
        }
        return months;
    }

    private WeatherByGeoIdData mapWeatherByGeoIdData(WeatherByGeoIdGraphQLData data) {
        return WeatherByGeoIdData.builder()
                .url(data.weatherByGeoId.url)
                .forecastItems(mapForecastDay(data.weatherByGeoId.forecast.days))
                .climateItems(mapClimateMonths(data.weatherByGeoId.climate.months))
                .build();
    }

    private List<WeatherByGeoIdData.ClimateItem> mapClimateMonths(List<WeatherByGeoIdGraphQLData.ClimateMonth> months) {
        List<WeatherByGeoIdData.ClimateItem> result = new LinkedList<>();
        var month = LocalDate.now().getMonth();
        for (WeatherByGeoIdGraphQLData.ClimateMonth climateMonth: months) {
            result.add(WeatherByGeoIdData.ClimateItem.builder()
                    .month(month)
                    .minNightTemperature(climateMonth.getMinNightTemperature())
                    .maxDayTemperature(climateMonth.getMaxDayTemperature())
                    .precipitationDays(climateMonth.getPrecDays())
                    .overcastDays(climateMonth.getOvercastDays())
                    .humidity(climateMonth.getHumidity())
                    .waterTemperature(climateMonth.getWaterTemperature())
                    .build());
            month = month.plus(1);
        }
        return result;
    }

    private List<WeatherByGeoIdData.ForecastItem> mapForecastDay(List<WeatherByGeoIdGraphQLData.ForecastDay> days) {
        List<WeatherByGeoIdData.ForecastItem> result = new LinkedList<>();

        var date = LocalDate.now();
        for (WeatherByGeoIdGraphQLData.ForecastDay day: days) {
            var item = day.summary.day;
            result.add(WeatherByGeoIdData.ForecastItem.builder()
                    .date(date)
                    .humidity(item.getHumidity())
                    .waterTemperature(item.getWaterTemperature())
                    .temperature(item.getTemperature())
                    .minTemperature(item.getMinTemperature())
                    .icon(item.getIcon())
                    .build());
            date = date.plusDays(1);
        }
        return result;
    }

    public <RC> CompletableFuture<RC> executeGraphQLRequest(String query, GraphQLRequestVariables variables, Class<RC> responseType) {
        var requestData = new GraphQLRequestData();
        requestData.setQuery(query);
        if (variables != null) {
            requestData.setVariables(variables);
        }

        return sendRequest(
                "POST",
                "graphql/query",
                requestData,
                responseType,
                "graphQL"
        );
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        RequestBuilder rb = super.createBaseRequestBuilder(method, path, body);
        if (tvmEnabled && tvm != null) {
            rb.addHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), tvm.getServiceTicket(tvmAlias));
        }
        rb.addHeader("Content-Type", "application/json");
        return rb;
    }
}
