package main

import (
	"context"
	"fmt"
	"os"
	"strings"
	"sync"

	"github.com/go-resty/resty/v2"
	"github.com/jonboulle/clockwork"
	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc/codes"
	ghealth "google.golang.org/grpc/health"
	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	aviaOrderAPI "a.yandex-team.ru/travel/app/backend/api/aviaorder/v1"
	aviaPriceAPI "a.yandex-team.ru/travel/app/backend/api/aviaprice/v1"
	calendarApi "a.yandex-team.ru/travel/app/backend/api/calendar/v1"
	geolocationAPI "a.yandex-team.ru/travel/app/backend/api/geolocation/v1"
	hotelFavoriteAPI "a.yandex-team.ru/travel/app/backend/api/hotelfavorites/v1"
	hotelOrderAPI "a.yandex-team.ru/travel/app/backend/api/hotelorder/v1"
	hotelReviewsAPI "a.yandex-team.ru/travel/app/backend/api/hotelreviews/v1"
	hotelsAPI "a.yandex-team.ru/travel/app/backend/api/hotels/v1"
	personalizationAPI "a.yandex-team.ru/travel/app/backend/api/personalization/v1"
	serverconfigAPI "a.yandex-team.ru/travel/app/backend/api/serverconfig/v1"
	systemAPI "a.yandex-team.ru/travel/app/backend/api/system/v1"
	systemLocalhostAPI "a.yandex-team.ru/travel/app/backend/api/systemlocalhost/v1"
	travelersAPI "a.yandex-team.ru/travel/app/backend/api/travelers/v1"
	"a.yandex-team.ru/travel/app/backend/internal/app"
	aviaHandler "a.yandex-team.ru/travel/app/backend/internal/avia/handler"
	aviaOrderHandler "a.yandex-team.ru/travel/app/backend/internal/avia/order/handler"
	aviaPriceHandler "a.yandex-team.ru/travel/app/backend/internal/avia/price/handler"
	"a.yandex-team.ru/travel/app/backend/internal/calendar"
	calendarHandler "a.yandex-team.ru/travel/app/backend/internal/calendar/handler"
	"a.yandex-team.ru/travel/app/backend/internal/cityimages"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	geolocationHandler "a.yandex-team.ru/travel/app/backend/internal/geolocation/handler"
	"a.yandex-team.ru/travel/app/backend/internal/healthswitch"
	"a.yandex-team.ru/travel/app/backend/internal/hotels"
	hotelFavoriteHandler "a.yandex-team.ru/travel/app/backend/internal/hotels/favorites/handler"
	hotelsHandler "a.yandex-team.ru/travel/app/backend/internal/hotels/handler"
	hotelOrderHandler "a.yandex-team.ru/travel/app/backend/internal/hotels/order/handler"
	hotelReviewsHandler "a.yandex-team.ru/travel/app/backend/internal/hotels/reviews/handler"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviabackendclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviapersonalizationclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviasuggestclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/calendarclient"
	contentadmin "a.yandex-team.ru/travel/app/backend/internal/lib/contentadminclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/exp3matcher"
	"a.yandex-team.ru/travel/app/backend/internal/lib/priceindexclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelersclient"
	personalizationHandler "a.yandex-team.ru/travel/app/backend/internal/personalization/handler"
	"a.yandex-team.ru/travel/app/backend/internal/points"
	"a.yandex-team.ru/travel/app/backend/internal/references"
	"a.yandex-team.ru/travel/app/backend/internal/serverconfig"
	serverConfigHandler "a.yandex-team.ru/travel/app/backend/internal/serverconfig/handler"
	systemHandler "a.yandex-team.ru/travel/app/backend/internal/system/handler"
	systemLocalhostHandler "a.yandex-team.ru/travel/app/backend/internal/systemlocalhost/handler"
	travelersService "a.yandex-team.ru/travel/app/backend/internal/travelers"
	travelersHandler "a.yandex-team.ru/travel/app/backend/internal/travelers/handler"
	"a.yandex-team.ru/travel/avia/library/go/services/featureflag"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/errorbooster"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	grpcmetrics "a.yandex-team.ru/travel/library/go/grpcutil/metrics"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/grpcutil/service/servingstatus"
	grpcTracing "a.yandex-team.ru/travel/library/go/grpcutil/tracing"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
)

const serviceName = "travel_app_backend"
const swaggerPrefix = "/api"

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		os.Exit(1)
	}

	errBoosterClient, err := errorbooster.NewDeployClient(
		app.Cfg.ErrorBooster,
		string(app.Cfg.EnvType),
	)
	if err != nil {
		fmt.Println("failed to create error booster client", err)
		ctxCancel()
		os.Exit(1)
	}

	app.Cfg.DetectEnvironmentSpecificSettings()
	logger, err := logging.NewLogger(
		logging.WithConfig(&app.Cfg.Logging),
		logging.WithFormat(logging.DeployFormat),
		logging.WithErrorBooster(errBoosterClient),
	)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		os.Exit(1)
	}
	needToLogRequestAndResponse := strings.EqualFold(app.Cfg.Logging.Level, "DEBUG")

	tracerCloser := tracing.InitializeGlobalTracer(serviceName, logger)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	var bbOpts []httpbb.Option
	tvmClient, err := tvmtool.NewDeployClient()
	if err != nil {
		logger.Fatal("failed to create tvm client", log.Error(err))
	}
	bbOpts = append(bbOpts, httpbb.WithTVM(tvmClient))
	bb, err := httpbb.NewClient(*app.Cfg.Auth.BlackBoxEnv, bbOpts...)
	if err != nil {
		logger.Fatal("failed to create blackbox client", log.Error(err))
	}

	rootRegistry := metrics.NewRegistryWithDeployTags()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	// geobase
	geoBase, err := geobase.NewGeobase(app.Cfg.Geobase, logger)
	if err != nil {
		logger.Fatal("failed to create geobase", log.Error(err))
	}
	defer geoBase.Destroy()

	// Health
	shutdownSwitch := healthswitch.NewSwitch()
	healthServer := ghealth.NewServer()
	servingStatusService := servingstatus.NewService(
		logger,
		serviceName,
		healthServer.SetServingStatus,
		app.Cfg.HealthCheck.UpdateInterval,
		clockwork.NewRealClock(),
	).Requires(
		func() bool {
			return shutdownSwitch.GetHealth()
		},
	)
	servingStatusService.MonitorServingStatus(ctx)
	l10nService, err := l10n.CreateAndLoad(app.Cfg.L10N, logger)
	if err != nil {
		logger.Fatal("unable to initialize l10n service", log.Error(err))
	}
	exp3client := exp3matcher.NewHTTPClient(app.Cfg.Exp3Matcher, logger, rootRegistry, l10nService, geoBase)
	serverConfigService, err := serverconfig.NewService(app.Cfg.ServerConfig, app.Cfg.EnvType, logger, exp3client)
	if err != nil {
		logger.Fatal("failed to create server config service", log.Error(err))
	}

	unifiedAgentClient, err := unifiedagent.NewGrpcClient(&app.Cfg.UnifiedClient, logger, nil)
	if err != nil {
		logger.Fatal("failed to create unified agent client", log.Error(err))
	}
	defer unifiedAgentClient.Close()

	// Общие справочники
	referencesRegistry, err := references.NewRegistry(app.Cfg.Dicts, logger)
	if err != nil {
		logger.Fatal("failed to create references registry", log.Error(err))
	}

	// grpc services
	srvCfgHandler := serverConfigHandler.NewGRPCServerConfigHandler(logger, serverConfigService, l10nService)
	system := systemHandler.NewGRPCSystemHandler(healthServer, serviceName, ctxCancel,
		app.Cfg.HealthCheck.ShutdownInterval, shutdownSwitch, app.Cfg.EnvType)

	systemLocalhost := systemLocalhostHandler.NewGRPCSystemLocalhostHandler(healthServer, serviceName, ctxCancel,
		app.Cfg.HealthCheck.ShutdownInterval, shutdownSwitch)
	systemLocalhost.SetReloadableServices(serverConfigService, l10nService, referencesRegistry)

	serviceTicketGetter := func(ctx context.Context) (string, error) {
		return tvmClient.GetServiceTicketForAlias(ctx, "travelers")
	}
	userTicketGetter := func(ctx context.Context) (string, error) {
		auth := common.GetAuth(ctx)
		if auth == nil {
			return "", fmt.Errorf("no user ticket")
		}
		return auth.UserTicket, nil
	}
	tc := travelersclient.NewHTTPClient(logger, &app.Cfg.Travelers, userTicketGetter, serviceTicketGetter,
		needToLogRequestAndResponse, rootRegistry)
	tCache, err := travelersService.NewTravelersCache(tc, travelersService.DefaultTravelersCacheConfig, logger)
	if err != nil {
		logger.Fatal("failed to create travelers cache", log.Error(err))
	}
	tAdminCache, err := travelersService.NewTravelersAdminCache(logger)
	if err != nil {
		logger.Fatal("failed to create travelers admin cache", log.Error(err))
	}
	travelers := travelersHandler.NewGRPCTravelersHandler(logger, tc, tCache, tAdminCache, l10nService)
	err = l10nService.Register(travelers)
	if err != nil {
		logger.Fatal("failed to register travelers in l10n service", log.Error(err))
	}

	travelAPIClient := travelapiclient.NewHTTPClient(
		logger,
		app.Cfg.EnvType,
		app.Cfg.TravelAPI,
		needToLogRequestAndResponse,
		rootRegistry,
		common.GetTVMHeaderProvider(tvmClient, "hotels"),
		common.GetAuthHeaders,
	)
	translateService := hotels.NewTranslationService(logger, l10nService)
	for keyset, keys := range translateService.GetKeysets() {
		err = l10nService.RegisterKeyset(keyset, keys)
		if err != nil {
			logger.Fatal("failed to register translateService in l10n service", log.Error(err))
		}
	}

	hotelsService := hotels.New(logger, travelAPIClient, app.Cfg.Hotels, translateService)
	hotelsGRPC := hotelsHandler.GRPCHotelsHandler{Service: hotelsService}

	hotelOrderService := hotelOrderHandler.NewGRPCHotelOrderHandler(logger, app.Cfg.EnvType, travelAPIClient, l10nService, &app.Cfg.Hotels.Order)
	err = l10nService.Register(hotelOrderService)
	if err != nil {
		logger.Fatal("failed to register hotel-order-service in l10n service", log.Error(err))
	}

	contentAdminClient, err := contentadmin.NewRetryableClient(logger, app.Cfg.Tvm.SelfAppID, app.Cfg.ContentAdmin, rootRegistry)
	if err != nil {
		logger.Fatal("failed to build content admin retryable client", log.Error(err))
	}
	cityImagesService := cityimages.NewService(logger, app.Cfg.CityImages, contentAdminClient)
	go cityImagesService.RunCaching(ctx)
	hotelFavoriteService := hotelFavoriteHandler.NewGRPCHotelFavoritesHandler(&app.Cfg.Hotels.Favorites, logger, travelAPIClient, cityImagesService, hotelsService)

	hotelReviewsService := hotelReviewsHandler.NewGRPCHandler(&app.Cfg.Hotels.Reviews, logger, travelAPIClient, translateService)

	aviaBackendClient := aviabackendclient.NewHTTPClient(logger, app.Cfg.AviaBackend, needToLogRequestAndResponse, rootRegistry)
	geolocation := geolocationHandler.NewGRPCGeolocationHandler(logger, geoBase, referencesRegistry, aviaBackendClient)

	featureFlagClient := featureflag.NewClient(app.Cfg.AviaFeatureFlagAPI.BaseURL, "travel-app-backend", logger, resty.New().SetTimeout(app.Cfg.AviaFeatureFlagAPI.Timeout).GetClient())
	featureFlagStorage := featureflag.NewStorage(featureFlagClient, app.Cfg.AviaFeatureFlagAPI.RefreshInterval, logger, clockwork.NewRealClock())
	featureFlagStorage.StartPeriodicUpdates()
	aviaTDAPIClient := aviatdapiclient.NewHTTPClient(logger, app.Cfg.AviaTDAPI, needToLogRequestAndResponse, rootRegistry)
	aviaSuggestClient := aviasuggestclient.NewHTTPClient(logger, app.Cfg.AviaSuggest, needToLogRequestAndResponse, rootRegistry)
	avia := aviaHandler.NewGRPCAviaHandler(logger, app.Cfg.EnvType, aviaTDAPIClient, aviaSuggestClient, app.Cfg.Avia, aviaBackendClient, featureFlagStorage, unifiedAgentClient, geoBase, referencesRegistry, exp3client)

	priceIndexClient := priceindexclient.NewHTTPClient(logger, &app.Cfg.PriceIndex, needToLogRequestAndResponse, rootRegistry)
	pointsParser := points.NewParser(referencesRegistry)
	aviaPrice := aviaPriceHandler.NewGRPCHandler(logger, &app.Cfg.Avia.Price, priceIndexClient, pointsParser)

	aviaOrderService := aviaOrderHandler.NewGRPCAviaOrderHandler(logger, travelAPIClient, l10nService, &app.Cfg.Avia.Order)
	err = l10nService.Register(aviaOrderService)
	if err != nil {
		logger.Fatal("failed to register avia-order-service in l10n service", log.Error(err))
	}

	aviaPersonalizationClient, err := aviapersonalizationclient.NewGRPCClient(
		ctx, logger, app.Cfg.AviaPersonalizationClient, rootRegistry,
	)
	if err != nil {
		logger.Fatal("failed to create aviaPersonalizationClient", log.Error(err))
	}
	personalization := personalizationHandler.NewGRPCPersonalizationHandler(logger, aviaPersonalizationClient, pointsParser)

	calendarClient := calendarclient.NewHTTPClient(logger, app.Cfg.CalendarClient, rootRegistry, common.GetTVMHeaderProvider(tvmClient, "calendar"))
	calendarService := calendar.NewService(logger, app.Cfg.Calendar, calendarClient)
	go calendarService.RunCaching(ctx)
	calendarGRPC := calendarHandler.NewGRPCCalendarHandler(calendarService)

	system.SetPingableServices(avia)

	grpcServiceRegisters := []grpcserver.ServiceRegisterer{
		srvCfgHandler.GetServiceRegisterer(),
		system.GetServiceRegisterer(),
		travelers.GetServiceRegisterer(),
		hotelsGRPC.GetServiceRegisterer(),
		hotelOrderService.GetServiceRegisterer(),
		hotelFavoriteService.GetServiceRegisterer(),
		hotelReviewsService.GetServiceRegisterer(),
		geolocation.GetServiceRegisterer(),
		avia.GetServiceRegisterer(),
		aviaOrderService.GetServiceRegisterer(),
		aviaPrice.GetServiceRegisterer(),
		calendarGRPC.GetServiceRegisterer(),
		personalization.GetServiceRegisterer(),
	}

	standardSpecs := []grpcgateway.HeaderSpec{
		common.SessionIDHTTPHeader,
		common.UserAgentHTTPHeader,
		common.AcceptLanguageHTTPHeader,
		common.UUIDHTTPHeader,
		common.RequestIDHTTPHeader,
		common.DeviceIDHTTPHeader,
	}
	standardSpecs = appendTracingHeaders(standardSpecs, app.Cfg.Tracing.TraceHeaders)

	grpcGateway := grpcgateway.NewGateway(&app.Cfg.GrpcGateway,
		grpcgateway.NewService("system_api", swaggerPrefix, app.Cfg.Grpc.Addr, systemAPI.RegisterSystemAPIHandlerFromEndpoint, nil),
		grpcgateway.NewService("server_config_api", swaggerPrefix, app.Cfg.Grpc.Addr,
			serverconfigAPI.RegisterServerConfigAPIHandlerFromEndpoint,
			standardSpecs),
		grpcgateway.NewService(
			"travelers_api",
			swaggerPrefix,
			app.Cfg.Grpc.Addr,
			travelersAPI.RegisterTravelersAPIHandlerFromEndpoint,
			standardSpecs),
		grpcgateway.NewService("hotels_api", swaggerPrefix, app.Cfg.Grpc.Addr,
			hotelsAPI.RegisterHotelsAPIHandlerFromEndpoint, standardSpecs),
		grpcgateway.NewService("hotel_order_api", swaggerPrefix, app.Cfg.Grpc.Addr,
			hotelOrderAPI.RegisterHotelOrderAPIHandlerFromEndpoint, standardSpecs),
		grpcgateway.NewService("hotel_favorites_api", swaggerPrefix, app.Cfg.Grpc.Addr,
			hotelFavoriteAPI.RegisterHotelFavoritesAPIHandlerFromEndpoint, standardSpecs),
		grpcgateway.NewService("hotel_reviews_api", swaggerPrefix, app.Cfg.Grpc.Addr,
			hotelReviewsAPI.RegisterHotelReviewsAPIHandlerFromEndpoint, standardSpecs),
		grpcgateway.NewService("geolocation_api", swaggerPrefix, app.Cfg.Grpc.Addr, geolocationAPI.RegisterGeolocationAPIHandlerFromEndpoint,
			appendTracingHeaders([]grpcgateway.HeaderSpec{common.RealIPHeader, common.AcceptLanguageHTTPHeader}, app.Cfg.Tracing.TraceHeaders)),
		grpcgateway.NewService("avia_api", swaggerPrefix, app.Cfg.Grpc.Addr, aviaAPI.RegisterAviaAPIHandlerFromEndpoint,
			appendTracingHeaders(
				[]grpcgateway.HeaderSpec{
					common.AcceptLanguageHTTPHeader,
					common.ABExperimentsHeader,
				},
				app.Cfg.Tracing.TraceHeaders,
			),
		),
		grpcgateway.NewService("avia_order_api", swaggerPrefix, app.Cfg.Grpc.Addr, aviaOrderAPI.RegisterAviaOrderAPIHandlerFromEndpoint,
			standardSpecs),
		grpcgateway.NewService("avia_price_api", swaggerPrefix, app.Cfg.Grpc.Addr, aviaPriceAPI.RegisterAviaPriceAPIHandlerFromEndpoint,
			standardSpecs),
		grpcgateway.NewService("calendar_api", swaggerPrefix, app.Cfg.Grpc.Addr,
			calendarApi.RegisterCalendarAPIHandlerFromEndpoint, standardSpecs),
		grpcgateway.NewService("personalization_api", swaggerPrefix, app.Cfg.Grpc.Addr,
			personalizationAPI.RegisterPersonalizationAPIHandlerFromEndpoint, standardSpecs),
	)

	localhostGrpcServiceRegisters := []grpcserver.ServiceRegisterer{
		systemLocalhost.GetServiceRegisterer(),
	}
	localhostGrpcGateway := grpcgateway.NewGateway(app.Cfg.LocalhostGrpcGateway.ToConfig(),
		grpcgateway.NewService("system_localhost_api", swaggerPrefix, app.Cfg.LocalhostGrpc.Addr, systemLocalhostAPI.RegisterSystemLocalhostAPIHandlerFromEndpoint, nil),
	)

	// run server
	wg := sync.WaitGroup{}
	wg.Add(7)
	go func() {
		defer wg.Done()
		tCache.RunUpdater()
		logger.Fatal("travelers cache updater exited")
	}()

	go func() {
		defer wg.Done()
		avia.RunRedisChecker(ctx)
		logger.Fatal("avia redis checker exited")
	}()

	runGrpcServer := func(serverName string, grpcConfig *grpcserver.GrpcConfig, grpcServiceRegisters []grpcserver.ServiceRegisterer) {
		defer wg.Done()

		filterFunc := func(ctx context.Context, fullMethodName string) bool {
			for _, skipVal := range app.Cfg.Tracing.SkipTracingMethodNameParts {
				if strings.Contains(fullMethodName, skipVal) {
					return false
				}
			}
			return true
		}
		filterFuncOpt := grpcTracing.WithFilterFunc(filterFunc)

		requestHandleFunc := func(ctx context.Context, req interface{}) {

			span := opentracing.SpanFromContext(ctx)
			if span == nil {
				return
			}

			md, ok := metadata.FromIncomingContext(ctx)
			if ok {
				for _, header := range app.Cfg.Tracing.TraceHeaders {
					tagName := strings.ToLower(header)
					if val, ok := md[tagName]; ok && len(val) > 0 {
						span = span.SetTag(tagName, val[0])
					}
				}
			}
		}
		handlerFuncOpt := grpcTracing.WithRequestHandlerFunc(requestHandleFunc)
		tracingInterceptor := grpcTracing.NewGrpcTracingInterceptor(filterFuncOpt, handlerFuncOpt)
		serverBuilder := grpcserver.NewCleanGrpcServerBuilder(*grpcConfig, logger).
			WithRegisterers(grpcServiceRegisters...).
			WithInterceptors(
				tracingInterceptor,
				grpcserver.NewGrpcLogFieldsInterceptor(grpcConfig.HeadersNameToLogNameMapping, true),
				grpcserver.DefaultRecoveryInterceptor(logger),
				grpcserver.DefaultLogInterceptor(logger),
				common.NewGrpcL10NInterceptor(logger, app.Cfg.L10N.SupportedLanguages, app.Cfg.L10N.SupportedCountries),
				common.NewABInterceptor(),
			).
			WithMetrics(
				rootRegistry.WithPrefix("grpc"),
				grpcmetrics.WithCodesCountPretty(
					codes.Canceled,
					codes.Unknown,
					codes.InvalidArgument,
					codes.DeadlineExceeded,
					codes.NotFound,
					codes.AlreadyExists,
					codes.PermissionDenied,
					codes.Unauthenticated,
					codes.ResourceExhausted,
					codes.FailedPrecondition,
					codes.Aborted,
					codes.OutOfRange,
					codes.Unimplemented,
					codes.Internal,
					codes.Unavailable,
					codes.DataLoss,
				)).
			WithHealthServer(healthServer)
		if bb != nil {
			serverBuilder = serverBuilder.WithAuth(common.BuildBlackBoxAuthFunction(app.Cfg.Auth, bb))
		}
		err := serverBuilder.Build().Run(ctx)
		if err != nil {
			msg := fmt.Sprintf("Error while running %s server", serverName)
			logger.Fatal(msg, log.Error(err))
		} else {
			logger.Infof("%s server is stopped", serverName)
		}
	}

	go runGrpcServer("GRPC", &app.Cfg.Grpc, grpcServiceRegisters)
	go runGrpcServer("localhost GRPC", app.Cfg.LocalhostGrpc.ToGrpcConfig(), localhostGrpcServiceRegisters)

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(ctx, app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while running metrics server", log.Error(err))
		} else {
			logger.Info("Metrics server is stopped")
		}
	}()

	runGrpcGateway := func(serverName string, gateway *grpcgateway.Gateway) {
		defer wg.Done()
		err = gateway.Run(ctx)
		if err != nil {
			msg := fmt.Sprintf("Error while running %s", serverName)
			logger.Fatal(msg, log.Error(err))
		} else {
			logger.Infof("%s server is stopped", serverName)
		}
	}

	go runGrpcGateway("GRPC-gateway", grpcGateway)
	go runGrpcGateway("localhost-GRPC-gateway", localhostGrpcGateway)

	logger.Info(
		"Started",
		log.String("http", app.Cfg.GrpcGateway.Address),
		log.String("grpc", app.Cfg.Grpc.Addr),
	)
	logger.Info(
		"Started localhost",
		log.String("http", app.Cfg.LocalhostGrpcGateway.Address),
		log.String("grpc", app.Cfg.LocalhostGrpc.Addr),
	)
	wg.Wait()
	logger.Info("All services are stopped")
}

func appendTracingHeaders(specs []grpcgateway.HeaderSpec, tracingHeaders []string) []grpcgateway.HeaderSpec {
	for _, val := range tracingHeaders {
		headerSpec := common.NewTracingHeader(val)
		specs = append(specs, headerSpec)
	}
	return specs
}
