package main

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"reflect"
	"time"

	"github.com/spf13/cobra"
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/app/backend/internal/lib/tankerclient"
	sandboxplanner "a.yandex-team.ru/travel/library/go/sandbox_planner"
)

func main() {
	if err := sandboxplanner.ReplaceArgsFromEnv(); err != nil {
		fmt.Println("can not replace args:", err)
		return
	}
	rootCMD := &cobra.Command{
		Use: "l10n-importer",
	}

	rootCMD.PersistentFlags().String("tanker-token", "", "OAuth token of a user having tanker access")
	rootCMD.PersistentFlags().String("tanker-project", "travel-app", "Tanker project to export")
	rootCMD.PersistentFlags().String("last-released-revision-file", "", "Path to the last released revision")
	rootCMD.PersistentFlags().String("output-file", "output.json", "File to save exported data")
	rootCMD.PersistentFlags().String("output-resources-file", "output-resources.yaml", "File to describe resources to export")
	rootCMD.PersistentFlags().String("release-to", "", "Environment to release resources to")
	_ = rootCMD.MarkPersistentFlagRequired("tanker-token")

	importCMD := &cobra.Command{
		Use:  "import",
		RunE: runImport,
	}

	rootCMD.AddCommand(importCMD)
	_ = rootCMD.Execute()

}

type resourceToCreate struct {
	Type        string                 `yaml:"type"`
	Description string                 `yaml:"description"`
	Path        string                 `yaml:"path"`
	Attrs       map[string]interface{} `yaml:"attrs"`
}

type releaseToMake struct {
	Env      string `yaml:"env"`
	Title    string `yaml:"title"`
	Comments string `yaml:"comments"`
}

type resources struct {
	Resources []resourceToCreate `yaml:"resources"`
	Release   *releaseToMake     `yaml:"release"`
}

func runImport(cmd *cobra.Command, args []string) error {
	startedAt := time.Now()
	logger, err := arczap.New(arczap.ConsoleConfig(log.InfoLevel))
	if err != nil {
		return xerrors.Errorf("unable to init config: %w", err)
	}
	project, err := cmd.Flags().GetString("tanker-project")
	if err != nil {
		return xerrors.Errorf("unable to get tanker-project: %w", err)
	}
	token, err := cmd.Flags().GetString("tanker-token")
	if err != nil {
		return xerrors.Errorf("unable to get tanker-token: %w", err)
	}
	outputFile, err := cmd.Flags().GetString("output-file")
	if err != nil {
		return xerrors.Errorf("unable to get output-file: %w", err)
	}
	outputResourcesFile, err := cmd.Flags().GetString("output-resources-file")
	if err != nil {
		return xerrors.Errorf("unable to get output-resource-file: %w", err)
	}
	lastRevFile, err := cmd.Flags().GetString("last-released-revision-file")
	if err != nil {
		return xerrors.Errorf("unable to get last-released-revision-file: %w", err)
	}
	releaseTo, err := cmd.Flags().GetString("release-to")
	if err != nil {
		return xerrors.Errorf("unable to get release-to: %w", err)
	}
	cfg := tankerclient.NewProductionConfig(project, token)
	client := tankerclient.NewHTTPClient(&cfg)
	resp, err := client.Export(context.Background())
	if err != nil {
		return err
	}
	jsonData, err := json.MarshalIndent(resp, "", " ")
	if err != nil {
		return err
	}
	err = ioutil.WriteFile(outputFile, jsonData, 0644)
	if err != nil {
		return err
	}

	var lastRev tankerclient.ExportResponse
	if lastRevFile != "" {
		lastRevData, err := ioutil.ReadFile(lastRevFile)
		if err != nil {
			return err
		}
		err = json.Unmarshal(lastRevData, &lastRev)
		if err != nil {
			return err
		}
	} else {
		logger.Warn("No last-revision file specified")
	}
	var export resources
	if reflect.DeepEqual(resp, &lastRev) {
		logger.Info("Nothing changed")
	} else {
		logger.Info("New data in tanker")
		export.Resources = []resourceToCreate{
			{
				Type:        "TRAVEL_APP_BACKEND_L10N_DATA",
				Description: fmt.Sprintf("l10n data exported at %s", startedAt.Format(time.RFC3339)),
				Path:        outputFile,
				Attrs:       map[string]interface{}{"ttl": 21},
			},
		}
		if releaseTo != "" {
			export.Release = &releaseToMake{
				Env:      releaseTo,
				Title:    fmt.Sprintf("L10NData as of %s", startedAt.Format(time.RFC3339)),
				Comments: "Tanker data changed",
			}
		}
	}
	if outputResourcesFile != "" {
		marshalled, err := yaml.Marshal(export)
		if err != nil {
			return err
		}
		if err := ioutil.WriteFile(outputResourcesFile, marshalled, 0644); err != nil {
			return err
		}
		logger.Info("Resources exported")
	}
	return nil
}
