package app

import (
	"time"

	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/travel/app/backend/internal/avia"
	"a.yandex-team.ru/travel/app/backend/internal/calendar"
	"a.yandex-team.ru/travel/app/backend/internal/cityimages"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/hotels"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviabackendclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviapersonalizationclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviasuggestclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/calendarclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/contentadminclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/exp3matcher"
	"a.yandex-team.ru/travel/app/backend/internal/lib/priceindexclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelersclient"
	"a.yandex-team.ru/travel/app/backend/internal/references"
	"a.yandex-team.ru/travel/app/backend/internal/serverconfig"
	"a.yandex-team.ru/travel/library/go/errorbooster"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	"a.yandex-team.ru/travel/library/go/grpcutil/headers"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
)

const (
	LogKeyTravelSessionID     = "travelSessionId"
	LogKeyTravelDeviceID      = "travelDeviceId"
	LogKeyTravelUUID          = "travelUuid"
	LogKeyTravelABExperiments = "travelABExperiments"
)

type TvmConfig struct {
	WhiteList []uint32 `config:"tvm-white-list"`
	SelfAppID uint32   `config:"selfapp_id,required" yaml:"self_app_id"`
}

type Config struct {
	Auth                      common.AuthConfig
	Avia                      avia.Config
	AviaBackend               aviabackendclient.Config         `yaml:"avia_backend"`
	AviaPersonalizationClient aviapersonalizationclient.Config `yaml:"aviapersonalizationclient"`
	AviaSuggest               aviasuggestclient.Config         `yaml:"avia_suggest"`
	AviaFeatureFlagAPI        FeatureFlagConfig                `yaml:"avia_feature_flag_api"`
	AviaTDAPI                 aviatdapiclient.Config           `yaml:"avia_td_api"`
	CalendarClient            calendarclient.Config            `yaml:"calendar_api"`
	Calendar                  calendar.Config
	ContentAdmin              contentadminclient.Config `yaml:"content_admin"`
	CityImages                cityimages.Config
	Dicts                     references.Config
	EnvType                   common.EnvType `config:"yenv-type,required" yaml:"yandex_environment_type"`
	ErrorBooster              errorbooster.DeployConfig
	Exp3Matcher               exp3matcher.Config `yaml:"exp3_matcher"`
	Geobase                   geobase.Config
	Grpc                      grpcserver.GrpcConfig
	GrpcGateway               grpcgateway.Config
	LocalhostGrpc             grpcserver.NoEnvGrpcConfig
	LocalhostGrpcGateway      grpcgateway.NoEnvConfig
	HealthCheck               HealthCheckConfig
	Hotels                    hotels.Config
	Logging                   logging.Config
	L10N                      l10n.Config
	Metrics                   metricsserver.MetricsConfig
	ServerConfig              serverconfig.Config
	Tracing                   TracingConfig
	TravelAPI                 travelapiclient.TravelAPIConfig `yaml:"travel_api"`
	Travelers                 travelersclient.Config
	Tvm                       TvmConfig
	PriceIndex                priceindexclient.Config   `yaml:"price_index"`
	UnifiedClient             unifiedagent.ClientConfig `yaml:"unified_client"`
}

type HealthCheckConfig struct {
	UpdateInterval   time.Duration `config:"health-check-update-interval"`
	ShutdownInterval time.Duration `config:"health-check-shutdown-interval"`
}

var DefaultHealthCheckConfig = HealthCheckConfig{
	UpdateInterval:   1 * time.Second,
	ShutdownInterval: 10 * time.Second,
}

var DefaultTvmConfig = TvmConfig{
	WhiteList: nil,
	SelfAppID: 2030815,
}

var DefaultL10nConfig = l10n.Config{
	SupportedLanguages: []string{"ru"},
	SupportedCountries: []string{"RU"},
}

type GeobaseConfig struct {
	Path string `config:"geobase-path"`
}

type TracingConfig struct {
	TraceHeaders               []string
	SkipTracingMethodNameParts []string
}

var DefaultTracingConfig = TracingConfig{
	TraceHeaders:               []string{"Jaeger-Debug-Id"},
	SkipTracingMethodNameParts: []string{"SystemAPI"},
}

type FeatureFlagConfig struct {
	Timeout         time.Duration `config:"feature-flag-api-timeout"`
	BaseURL         string        `config:"feature-flag-api-baseurl,required"`
	RefreshInterval time.Duration `config:"feature-flag-api-refresh-interval"`
}

var DefaultFeatureFlagConfig = FeatureFlagConfig{
	Timeout:         10 * time.Second,
	BaseURL:         "https://feature-flag.testing.yandex-team.ru",
	RefreshInterval: 5 * time.Minute,
}

var Cfg = Config{
	Avia:                      avia.DefaultConfig,
	AviaBackend:               aviabackendclient.DefaultConfig,
	AviaPersonalizationClient: aviapersonalizationclient.DefaultConfig,
	AviaSuggest:               aviasuggestclient.DefaultConfig,
	AviaFeatureFlagAPI:        DefaultFeatureFlagConfig,
	AviaTDAPI:                 aviatdapiclient.DefaultConfig,
	CalendarClient:            calendarclient.DefaultConfig,
	Calendar:                  calendar.DefaultConfig,
	ContentAdmin:              contentadminclient.DefaultConfig,
	CityImages:                cityimages.DefaultConfig,
	Dicts:                     references.DefaultConfig,
	EnvType:                   common.DevelopmentEnv,
	ErrorBooster: errorbooster.DeployConfig{
		ServiceName:    "app.backend",
		RequestTimeout: 1 * time.Second,
	},
	Exp3Matcher: exp3matcher.DefaultConfig,
	Geobase:     geobase.DefaultConfig,
	Grpc: grpcserver.GrpcConfig{
		Addr:            "[::]:9001",
		AllowReflection: true,
		HeadersNameToLogNameMapping: map[string]string{
			headers.RealIP:                 logging.KeyIPHeader,
			headers.RequestID:              logging.KeyRequestID,
			headers.ForwardedHost:          logging.KeyHost,
			common.UserAgentGrpcHeader:     logging.KeyUserAgent,
			common.SessionIDGrpcHeader:     LogKeyTravelSessionID,
			common.DeviceIDGrpcHeader:      LogKeyTravelDeviceID,
			common.UUIDGrpcHeader:          LogKeyTravelUUID,
			common.ABExperimentsGrpcHeader: LogKeyTravelABExperiments,
		},
	},
	GrpcGateway: grpcgateway.Config{
		Enabled:       true,
		Address:       "[::]:80",
		EnableBinary:  true,
		EnableSwagger: true,
	},
	LocalhostGrpc: grpcserver.NoEnvGrpcConfig{
		Addr:                        "localhost:10001",
		AllowReflection:             true,
		HeadersNameToLogNameMapping: nil,
	},
	LocalhostGrpcGateway: grpcgateway.NoEnvConfig{
		Enabled:       true,
		Address:       "localhost:10080",
		EnableBinary:  true,
		EnableSwagger: true,
	},
	HealthCheck: DefaultHealthCheckConfig,
	Hotels:      hotels.DefaultConfig,
	L10N:        DefaultL10nConfig,
	Logging: logging.Config{
		Level:             "info",
		ErrorBoosterLevel: "error",
		ErrorBoosterAdditionalFields: []string{
			LogKeyTravelSessionID,
			LogKeyTravelDeviceID,
			LogKeyTravelUUID,
			LogKeyTravelABExperiments,
		},
	},
	Metrics:       metricsserver.DefaultMetricsConfig,
	Tracing:       DefaultTracingConfig,
	TravelAPI:     travelapiclient.DefaultTravelAPIConfig,
	Travelers:     travelersclient.DefaultTravelersConfig,
	Tvm:           DefaultTvmConfig,
	PriceIndex:    priceindexclient.DefaultConfig,
	UnifiedClient: unifiedagent.DefaultClientConfig,
}

func (c *Config) DetectEnvironmentSpecificSettings() {
	if c.Auth.BlackBoxEnv == nil {
		switch c.EnvType {
		case common.DevelopmentEnv:
			c.Auth.BlackBoxEnv = &httpbb.TestEnvironment
		case common.TestingEnv:
			c.Auth.BlackBoxEnv = &httpbb.TestEnvironment
		case common.ProductionEnv:
			c.Auth.BlackBoxEnv = &httpbb.ProdEnvironment
		}
	}
}
