package avia

import (
	"time"

	"a.yandex-team.ru/travel/app/backend/internal/avia/search"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/sorting"
	exp3pb "a.yandex-team.ru/travel/app/backend/internal/lib/exp3matcher/proto/v1"
)

type Config struct {
	Search            SearchConfig `config:"search,required"`
	Redis             RedisConfig  `config:"redis,required"`
	Order             OrderConfig
	Price             PriceConfig
	Sorter            sorting.ManagerSorterConfig
	Exp3DefaultConfig *exp3pb.GetAviaConfigRspData
}

type SearchConfig struct {
	Cache            search.AviaAppRedisCacheConfig
	CacheFormatTools search.CacheFormatToolsConfig
}

type OrderConfig struct {
	HappyPageImageSuccessURL string
	HappyPageImageFailedURL  string
}

type PriceConfig struct {
	// Число дней для запроса цен от даты вылета
	CountDays               int
	MedianCoefficient       float64
	PriceExpirationInterval time.Duration
}

type RedisConfig struct {
	Addresses               []string      `config:"addresses,required"`
	MasterName              string        `config:"master_name,required" yaml:"master_name"`
	Password                string        `config:"redis_password,required"`
	DialTimeout             time.Duration `config:"dial_timeout"`
	ReadTimeout             time.Duration `config:"read_timeout"`
	WriteTimeout            time.Duration `config:"write_timeout"`
	MetricsBucketsExpStart  time.Duration `config:"metrics_bucket_start"`
	MetricsBucketsExpFactor float64       `config:"metrics_bucket_factor"`
	MetricsBucketsExpN      int           `config:"metrics_bucket_n"`
	CheckAliveInterval      time.Duration `config:"check_alive_interval"`
}

var DefaultConfig = Config{
	Search: SearchConfig{
		Cache:            search.DefaultAviaAppRedisCacheConfig,
		CacheFormatTools: search.DefaultCacheFormatToolsConfig,
	},
	Redis: RedisConfig{
		DialTimeout:             100 * time.Millisecond,
		ReadTimeout:             time.Second,
		WriteTimeout:            time.Second,
		MetricsBucketsExpStart:  10 * time.Millisecond,
		MetricsBucketsExpFactor: 1.2,
		MetricsBucketsExpN:      20,
		CheckAliveInterval:      time.Second,
	},
	Order: OrderConfig{
		HappyPageImageSuccessURL: "https://avatars.mds.yandex.net/get-travel-app/6353118/2a000001812597801f2cbdb5e703734f1a64/orig",
		HappyPageImageFailedURL:  "https://avatars.mds.yandex.net/get-travel-app/6058370/2a0000018125ad3bdd28bbc4ef85dcfa64c8/orig",
	},
	Price: PriceConfig{
		CountDays:               120,
		MedianCoefficient:       0.8,
		PriceExpirationInterval: 4 * time.Hour,
	},
	Sorter: sorting.ManagerSorterDefaultConfig,
	Exp3DefaultConfig: &exp3pb.GetAviaConfigRspData{
		TdApiConfig: &exp3pb.TdApiConfig{
			InstantSearchEnabled: true,
		},
	},
}
