package handler

import (
	"context"
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"strings"
	"sync/atomic"
	"time"

	"github.com/go-redis/redis/v8"
	"google.golang.org/genproto/googleapis/type/date"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia"
	"a.yandex-team.ru/travel/app/backend/internal/avia/logs"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
	"a.yandex-team.ru/travel/app/backend/internal/aviacommon"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviasuggestclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	exp3pb "a.yandex-team.ru/travel/app/backend/internal/lib/exp3matcher/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/lib/priceindexclient"
	"a.yandex-team.ru/travel/app/backend/internal/references"
	"a.yandex-team.ru/travel/avia/library/go/services/featureflag"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/library/go/redislib"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
)

const (
	serviceID    = "avia"
	tdAPITitle   = "avia ticket-daemon-api"
	suggestTitle = "avia suggests"
)

type AviaConfigGetter interface {
	GetAviaConfig(ctx context.Context) *exp3pb.GetAviaConfigRspData
}

type GRPCAviaHandler struct {
	logger             log.Logger
	aviaTDAPIClient    TDAPIClient
	aviaSuggestClient  SuggestClient
	redisClient        redis.UniversalClient
	config             avia.Config
	redisAlive         atomic.Value
	serviceSearch      *search.ServiceSearch
	priceIndexClient   priceindexclient.Client
	cache              *search.AviaAppRedisCache
	cacheFormatTools   *search.CacheFormatTools
	aviaConfigGetter   AviaConfigGetter
	aviaUsersSearchLog *logs.AviaUsersSearchLogLogger
}

func NewGRPCAviaHandler(
	logger log.Logger,
	env common.EnvType,
	aviaTDAPIClient TDAPIClient,
	aviaSuggestClient SuggestClient,
	config avia.Config,
	aviaBackendClient search.BackendClient,
	featureFlag featureflag.StorageInterface,
	uaClient unifiedagent.Client,
	geoBase geobase.Geobase,
	pointRegistry *references.Registry,
	aviaConfigGetter AviaConfigGetter,
) *GRPCAviaHandler {
	redisOptions := redis.UniversalOptions{
		Addrs:          config.Redis.Addresses,
		Password:       config.Redis.Password,
		DialTimeout:    config.Redis.DialTimeout,
		ReadTimeout:    config.Redis.ReadTimeout,
		WriteTimeout:   config.Redis.WriteTimeout,
		ReadOnly:       false,
		RouteByLatency: false,
		RouteRandomly:  false,
		MasterName:     config.Redis.MasterName,
	}
	redisClient := redis.NewUniversalClient(&redisOptions)
	redisClient.AddHook(redislib.NewMetricsHook(
		redislib.Prefix("avia-search-cache"),
		redislib.TimingsBuckets(metrics.MakeExponentialDurationBuckets(
			config.Redis.MetricsBucketsExpStart,
			config.Redis.MetricsBucketsExpFactor,
			config.Redis.MetricsBucketsExpN,
		)),
	))
	searchService := search.NewServiceSearch(logger)

	h := GRPCAviaHandler{
		logger:            logger.WithName(serviceID),
		aviaTDAPIClient:   aviaTDAPIClient,
		aviaSuggestClient: aviaSuggestClient,
		redisClient:       redisClient,
		config:            config,
		serviceSearch:     searchService,
		cache: search.NewAviaAppRedisCache(
			&config.Search.Cache,
			string(env),
			logger,
			redisClient,
			searchService,
		),
		cacheFormatTools: search.NewCacheFormatTools(
			&config.Search.CacheFormatTools,
			&config.Search.Cache,
			searchService,
			aviaBackendClient,
			featureFlag,
			&config.Sorter,
			logger,
		),
		aviaConfigGetter:   aviaConfigGetter,
		aviaUsersSearchLog: logs.NewAviaUsersSearchLogLogger(logger, uaClient, geoBase, pointRegistry),
	}
	h.redisAlive.Store(false)
	return &h
}

func (h *GRPCAviaHandler) ID() string {
	return serviceID
}

func (h *GRPCAviaHandler) Ping(ctx context.Context) error {
	redisAlive := h.redisAlive.Load().(bool)
	if !redisAlive {
		return fmt.Errorf("redis is not alive")
	}
	return nil
}

func (h *GRPCAviaHandler) checkRedis(ctx context.Context) {
	err := h.redisClient.Ping(context.Background()).Err()
	if err != nil {
		ctxlog.Error(ctx, h.logger, "ping redis error", log.Error(err))
	}
	h.redisAlive.Store(err == nil)
}

func (h *GRPCAviaHandler) RunRedisChecker(ctx context.Context) {
	t := time.NewTicker(h.config.Redis.CheckAliveInterval)
	defer t.Stop()
	for range t.C {
		h.checkRedis(ctx)
	}
}

func (h *GRPCAviaHandler) Suggest(ctx context.Context, request *aviaAPI.SuggestReq) (*aviaAPI.SuggestRsp, error) {
	locale := common.GetLocale(ctx)
	fieldType, err := convertFieldTypeToProto(request.FieldType)
	if err != nil {
		ctxlog.Error(ctx, h.logger, suggestTitle+" bad request: invalid field_type:", log.Error(err))
		return nil, status.Error(codes.InvalidArgument, suggestTitle+" bad request: invalid field_type")
	}
	response, err := h.aviaSuggestClient.Suggest(
		ctx,
		aviacommon.GetNationalVersionByCountryCode(locale.CountryCodeAlpha2),
		locale.Language,
		fieldType,
		request.Query,
		request.OtherQuery,
		request.OtherPointKey,
	)
	if err != nil {
		msg := suggestTitle + " invalid response"
		ctxlog.Error(ctx, h.logger, msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}
	result, err := convertSuggestsToProto(response)
	if err != nil {
		msg := suggestTitle + " invalid response"
		ctxlog.Error(ctx, h.logger, msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}
	return result, nil
}

func convertSuggestsToProto(response *aviasuggestclient.SuggestResponse) (*aviaAPI.SuggestRsp, error) {
	result := aviaAPI.SuggestRsp{
		Query: response.Query,
	}
	suggests, err := fillSuggests(response.Suggests)
	if err != nil {
		return nil, xerrors.Errorf("can't convert %s response", suggestTitle)
	}
	result.Suggests = suggests
	return &result, nil
}

func fillSuggests(suggests []aviasuggestclient.Suggest) ([]*aviaAPI.Suggest, error) {
	result := make([]*aviaAPI.Suggest, 0, len(suggests))
	for _, suggest := range suggests {
		pointType, err := common.GetPointType(suggest.PointKey)
		if err != nil {
			return nil, xerrors.Errorf("unknown point_type in '%s'", suggest.PointKey)
		}
		nested, err := fillSuggests(suggest.Nested)
		if err != nil {
			return nil, xerrors.Errorf("can't convert %s response", suggestTitle)
		}
		resultSuggest := aviaAPI.Suggest{
			Level:          uint32(suggest.Level),
			Title:          suggest.Title,
			PointKey:       suggest.PointKey,
			PointType:      pointType,
			PointAviaCode:  suggest.PointCode,
			RegionTitle:    suggest.RegionTitle,
			CityTitle:      suggest.CityTitle,
			CountryTitle:   suggest.CountryTitle,
			Misprint:       uint32(suggest.Missprint),
			HaveAirport:    suggest.HaveAirport.GetValue(),
			NestedSuggests: nested,
		}
		if resultSuggest.PointType == commonAPI.GeoPointType_GEO_POINT_TYPE_COUNTRY {
			// Из саджестов авиа приходит двухбуквенный код страны ISO 3166-1 но почему-то строчными буквами
			resultSuggest.PointAviaCode = strings.ToUpper(resultSuggest.PointAviaCode)
		}
		result = append(result, &resultSuggest)
	}
	return result, nil
}

func convertFieldTypeToProto(fieldType aviaAPI.SuggestFieldType) (aviasuggestclient.FieldType, error) {
	switch fieldType {
	case aviaAPI.SuggestFieldType_SUGGEST_FIELD_TYPE_FROM:
		return aviasuggestclient.FieldFrom, nil
	case aviaAPI.SuggestFieldType_SUGGEST_FIELD_TYPE_TO:
		return aviasuggestclient.FieldTo, nil
	}
	return "", xerrors.Errorf("unknown field_type %s", string(fieldType))
}

func (h *GRPCAviaHandler) InitSearch(ctx context.Context, req *aviaAPI.InitSearchReq) (*aviaAPI.InitSearchRsp, error) {

	locale := common.GetLocale(ctx)
	forward := dateFromProto(req.DateForward)
	if forward == nil {
		return nil, status.Error(codes.InvalidArgument, "bad request, empty date_forward")
	}
	rsp, err := h.aviaTDAPIClient.InitSearch(
		ctx,
		aviacommon.GetNationalVersionByCountryCode(locale.CountryCodeAlpha2),
		locale.Language,
		req.Passengers.Adults,
		req.Passengers.Children,
		req.Passengers.Infants,
		*forward,
		dateFromProto(req.DateBackward),
		serviceClassFromProto(req.ServiceClass),
		req.PointKeyFrom,
		req.PointKeyTo,
	)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, tdAPITitle)
	}
	if rsp.Status != aviatdapiclient.SuccessResponseStatus {
		return nil, status.Error(codes.Unknown, fmt.Sprintf("%s error response, status: %s", tdAPITitle, string(rsp.Status)))
	}

	h.aviaUsersSearchLog.Log(ctx, rsp.Data.QID)

	return &aviaAPI.InitSearchRsp{
		Qid: rsp.Data.QID,
	}, nil
}

func (h *GRPCAviaHandler) searchResultInternal(ctx context.Context, qid string, sort aviaAPI.SearchSort, selectedFilters *aviaAPI.SearchFiltersReq, onlyFilters bool) (*aviaAPI.SearchResultRsp, error) {
	ctx = ctxlog.WithFields(ctx, log.String("qid", qid))
	exp3aviaConfig := h.aviaConfigGetter.GetAviaConfig(ctx)
	if exp3aviaConfig == nil {
		exp3aviaConfig = h.config.Exp3DefaultConfig
	}

	err := filtering.CheckPreconditions(selectedFilters)
	if err != nil {
		msg := "filter preconditions error"
		ctxlog.Error(ctx, h.logger, msg, log.Error(err))
		return nil, status.Error(codes.FailedPrecondition, fmt.Sprintf("%s: %s", msg, err.Error()))
	}
	cacheResult, err := h.cache.Get(ctx, qid, sort, selectedFilters)
	if err != nil {
		ctxlog.Error(ctx, h.logger, "Error while getting data from cache", log.Error(err))
	} else if cacheResult != nil && cacheResult.Progress.Current == cacheResult.Progress.Total {
		response, err := h.cacheFormatTools.BuildResponseFromCacheData(ctx, qid, cacheResult, sort, selectedFilters, onlyFilters)
		if err != nil {
			ctxlog.Error(ctx, h.logger, "Error while generating response from cache", log.Error(err))
		} else {
			ctxlog.Debug(ctx, h.logger, fmt.Sprintf("filters in request: %s\nfilters in response: %s", prettyPrint(selectedFilters), prettyPrint(response.Filters)))
			return response, nil
		}
	}

	data, err := h.TryGetNewDataAndUpdateCache(ctx, qid, cacheResult, exp3aviaConfig)
	if err != nil {
		ctxlog.Error(ctx, h.logger, "Error while try to get new data  and update cache", log.Error(err))
		if _, isGrpcError := status.FromError(err); isGrpcError {
			return nil, err
		} else {
			return nil, status.Error(codes.Unknown, "No result data")
		}
	}

	response, err := h.cacheFormatTools.BuildResponseFromCacheData(ctx, qid, data, sort, selectedFilters, onlyFilters)
	if err != nil {
		ctxlog.Error(ctx, h.logger, "Error while generating response from cache", log.Error(err))
		return nil, status.Error(codes.Unknown, "No result data")
	}

	ctxlog.Debug(ctx, h.logger, fmt.Sprintf("filters in request: %s\nfilters in response: %s", prettyPrint(selectedFilters), prettyPrint(response.Filters)))
	return response, nil
}

func (h *GRPCAviaHandler) TryGetNewDataAndUpdateCache(
	ctx context.Context,
	qid string,
	cacheResult *aviaSearchProto.SearchResult,
	exp3aviaConfig *exp3pb.GetAviaConfigRspData,
) (*aviaSearchProto.SearchResult, error) {
	rsp, err := h.aviaTDAPIClient.SearchResult(ctx, qid, exp3aviaConfig.TdApiConfig)
	if err != nil {
		var netError net.Error
		var clientsCommonError clientscommon.StatusError
		if (xerrors.As(err, &netError) && netError.Timeout()) || (xerrors.As(err, &clientsCommonError) && (clientsCommonError.Status == http.StatusGatewayTimeout || clientsCommonError.Status == http.StatusRequestTimeout)) {
			// Если таймаут, то возвращаем что есть
			if cacheResult != nil {
				ctxlog.Error(ctx, h.logger, "return previous result", log.Error(err))
				return cacheResult, nil
			}
			// если еще ничего нет, то возвращаем пустые данные
			ctxlog.Error(ctx, h.logger, "return fake empty result", log.Error(err))
			return h.createEmptyCacheResult(qid)
		}
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, tdAPITitle)
	}
	if rsp.Status != aviatdapiclient.SuccessResponseStatus {
		return nil, status.Error(codes.Unknown, fmt.Sprintf("%s error response, status: %s", tdAPITitle, string(rsp.Status)))
	}

	cacheValue, err := h.cacheFormatTools.BuildCacheData(ctx, qid, rsp.Data)
	if err != nil {
		ctxlog.Error(ctx, h.logger, "Error while setting cache value", log.Error(err))
		return nil, status.Error(codes.Unknown, "Error while generating answer")
	}

	data, err := h.cache.Set(ctx, qid, cacheValue)
	if err != nil {
		ctxlog.Error(ctx, h.logger, "Error while setting cache value", log.Error(err))
	}
	if data == nil {
		return nil, status.Error(codes.Unknown, "No result data")
	}
	return data, nil
}

func (h *GRPCAviaHandler) SearchResult(ctx context.Context, req *aviaAPI.SearchResultReq) (*aviaAPI.SearchResultRsp, error) {
	return h.searchResultInternal(ctx, req.Qid, req.Sort, req.SelectedFilters, false)
}

func (h *GRPCAviaHandler) UpdateFilters(ctx context.Context, req *aviaAPI.UpdateFiltersReq) (*aviaAPI.UpdateFiltersRsp, error) {
	rsp, err := h.searchResultInternal(ctx, req.Qid, aviaAPI.SearchSort_SEARCH_SORT_UNKNOWN, req.SelectedFilters, true)
	if err != nil {
		return nil, err
	}

	return &aviaAPI.UpdateFiltersRsp{
		Filters:   rsp.Filters,
		Reference: rsp.Reference,
	}, nil
}

func (h *GRPCAviaHandler) createEmptyCacheResult(qid string) (*aviaSearchProto.SearchResult, error) {
	searchContext, err := searchcommon.ParseQIDToProto(qid)
	if err != nil {
		return nil, err
	}

	return &aviaSearchProto.SearchResult{
		Version: h.config.Search.Cache.Version,
		Progress: &aviaSearchProto.Progress{
			Current: 0,
			Total:   100,
		},
		Reference: &aviaSearchProto.Reference{
			Flights:       nil,
			Partners:      nil,
			Settlements:   nil,
			Stations:      nil,
			AviaCompanies: nil,
			Alliances:     nil,
		},
		Snippets:                 nil,
		ExpiresAt:                timestamppb.New(time.Now().Add(h.config.Search.Cache.CacheTTL)),
		SortedByRecommendedFirst: nil,
		SortedByCheapestFirst:    nil,
		SortedByExpensiveFirst:   nil,
		SortedByDeparture:        nil,
		SortedByArrival:          nil,
		SearchContext:            searchContext,
	}, nil
}

func (h *GRPCAviaHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		aviaAPI.RegisterAviaAPIServer(server, h)
	}
}

func serviceClassFromProto(class aviaAPI.ServiceClass) aviatdapiclient.ServiceClass {
	switch class {
	case aviaAPI.ServiceClass_SERVICE_CLASS_ECONOMY:
		return aviatdapiclient.EconomyServiceClass
	case aviaAPI.ServiceClass_SERVICE_CLASS_BUSINESS:
		return aviatdapiclient.BusinessServiceClass
	default:
		return aviatdapiclient.EconomyServiceClass
	}
}

func dateFromProto(day *date.Date) *aviatdapiclient.Date {
	if day == nil {
		return nil
	}
	return &aviatdapiclient.Date{Time: time.Date(int(day.Year), time.Month(day.Month), int(day.Day), 0, 0, 0, 0, time.UTC)}
}

func prettyPrint(i interface{}) string {
	s, _ := json.MarshalIndent(i, "", "  ")
	return string(s)
}

type TDAPIClient interface {
	InitSearch(
		ctx context.Context,
		nationalVersion string,
		lang string,
		adults uint32,
		children uint32,
		infants uint32,
		dateForward aviatdapiclient.Date,
		dateBackward *aviatdapiclient.Date,
		serviceClass aviatdapiclient.ServiceClass,
		pointFrom string,
		pointTo string,
	) (*aviatdapiclient.InitSearchRsp, error)

	SearchResult(ctx context.Context, qid string, exp3tdAPIConfig *exp3pb.TdApiConfig) (*aviatdapiclient.SearchResultRsp, error)
}

type SuggestClient interface {
	Suggest(
		ctx context.Context,
		nationalVersion string,
		lang string,
		field aviasuggestclient.FieldType,
		query string,
		otherQuery string,
		otherPoint string,
	) (*aviasuggestclient.SuggestResponse, error)
}
