package logs

import (
	"context"
	"fmt"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/library/go/core/log"
	uapb "a.yandex-team.ru/logbroker/unified_agent/plugins/grpc_input/proto"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/references"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/library/go/unifiedagent"
)

type BaseLogger struct {
	logger        log.Logger
	uaClient      unifiedagent.Client
	topic         string
	geoBase       geobase.Geobase
	pointRegistry *references.Registry
}

func NewBaseLogger(
	logger log.Logger,
	uaClient unifiedagent.Client,
	topic string,
	geoBase geobase.Geobase,
	pointRegistry *references.Registry,
) *BaseLogger {
	return &BaseLogger{
		logger:        logger,
		uaClient:      uaClient,
		topic:         topic,
		geoBase:       geoBase,
		pointRegistry: pointRegistry,
	}
}

func (l *BaseLogger) SetQidData(logElement map[string]interface{}, qid string) map[string]interface{} {
	parsedQid, err := searchcontext.ParseQID(qid)
	if err != nil {
		l.logger.Error(fmt.Sprintf("Can't parse QID: %s", qid), log.Error(err))
		return logElement
	}

	qKey, err := searchcontext.ExtractRawQKeyFromQID(qid)
	if err != nil {
		l.logger.Error(fmt.Sprintf("Can't parse QKey by QID: %s", qid), log.Error(err))
		qKey = ""
	}

	logElement["qid"] = qid
	logElement["when"] = parsedQid.QKey.DateForward.Format(timeformats.RFC3339Date)
	logElement["return_date"] = parsedQid.QKey.DateBackward.Format(timeformats.RFC3339Date)
	logElement["adult_seats"] = parsedQid.QKey.Adults
	logElement["children_seats"] = parsedQid.QKey.Children
	logElement["infant_seats"] = parsedQid.QKey.Infants
	logElement["fromId"] = parsedQid.QKey.PointFromKey
	logElement["toId"] = parsedQid.QKey.PointToKey
	logElement["lang"] = parsedQid.Lang
	logElement["national_version"] = parsedQid.QKey.NationalVersion
	logElement["klass"] = parsedQid.QKey.Class
	logElement["queryKey"] = qKey

	return logElement
}

func (l *BaseLogger) SetRealIPData(ctx context.Context, logElement map[string]interface{}) map[string]interface{} {
	if value := common.GetRealIP(ctx); value != nil {
		logElement["userip"] = *value

		region, err := l.geoBase.GetRegionByIP(*value)
		if err != nil {
			l.logger.Error(fmt.Sprintf("Can't get region by IP: %s", *value), log.Error(err))
			return logElement
		} else if region == nil {
			return logElement
		}

		logElement["user_from_geo_id"] = int64(region.ID)

		pointKey, exists := l.pointRegistry.GetPointKeyByGeoID(int(region.ID))
		if exists {
			logElement["user_from_key"] = pointKey
		}
	}

	return logElement
}

func (l *BaseLogger) SetPlatformData(ctx context.Context, logElement map[string]interface{}) map[string]interface{} {
	ua := common.GetUserAgent(ctx)
	pp, exists := ua.KnownOS.Pp()
	if exists {
		logElement["pp"] = int64(pp)
	}
	return logElement
}

func (l *BaseLogger) Log(data map[string]interface{}) {
	go func() {
		err := l.uaClient.SendJSON(data, []*uapb.Request_MessageMetaItem{{
			Key:   "log_name",
			Value: []string{l.topic},
		}})
		if err != nil {
			l.logger.Error(fmt.Sprintf("failed to send %s record to UnifiedAgent", l.topic), log.Error(err))
		}
	}()
}
