package handler

import (
	"context"
	"errors"
	"fmt"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/ptr"
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	aviaOrderAPI "a.yandex-team.ru/travel/app/backend/api/aviaorder/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	commonOrderAPI "a.yandex-team.ru/travel/app/backend/api/commonorder/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/l10n"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	"a.yandex-team.ru/travel/app/backend/internal/lib/travelapiclient/models"
)

const serviceName = "avia order"
const l10nKeyset = "backend_avia_order"
const purchaseTokenKey = "purchase_token"

type translationKey string

const (
	defaultWaitingTitleKey   translationKey = "workflow_default_waiting_title"
	defaultWaitingMessageKey translationKey = "workflow_default_waiting_message"
	preparingOrderTitleKey   translationKey = "workflow_preparing_order_title"
	preparingOrderMessageKey translationKey = "workflow_preparing_order_message"

	hpButtonTextKey translationKey = "workflow_hp_button_text"

	hpResultCellPrimaryTextSuccessKey          translationKey = "workflow_hp_result_cell_primary_text_success"
	hpResultCellPrimaryTextPaymentFailedKey    translationKey = "workflow_hp_result_cell_primary_text_payment_failed"
	hpResultCellPrimaryTextBookingCancelledKey translationKey = "workflow_hp_result_cell_primary_text_booking_cancelled"
	hpResultCellPrimaryTextUnknownKey          translationKey = "workflow_hp_result_cell_primary_text_unknown"

	hpResultCellSecondaryTextSuccessKey          translationKey = "workflow_hp_result_cell_secondary_text_success"
	hpResultCellSecondaryTextPaymentFailedKey    translationKey = "workflow_hp_result_cell_secondary_text_payment_failed"
	hpResultCellSecondaryTextBookingCancelledKey translationKey = "workflow_hp_result_cell_secondary_text_booking_cancelled"
	hpResultCellSecondaryTextUnknownKey          translationKey = "workflow_hp_result_cell_secondary_text_unknown"

	hpResultCellWhatsappTextKey  translationKey = "workflow_hp_result_cell_whatsapp_text"
	hpResultCellCallDataTextKey  translationKey = "workflow_hp_result_cell_call_data_text"
	hpResultCellCallDataPhoneKey translationKey = "workflow_hp_result_cell_call_data_phone"

	hpSupportCellCallDataTextKey  translationKey = "workflow_hp_support_cell_call_data_text"
	hpSupportCellCallDataPhoneKey translationKey = "workflow_hp_support_cell_call_data_phone"
	hpSupportCellPrimaryTextKey   translationKey = "workflow_hp_support_cell_primary_text"
	hpSupportCellSecondaryTextKey translationKey = "workflow_hp_support_cell_secondary_text"
)

type GRPCHandler struct {
	config          *avia.OrderConfig
	logger          log.Logger
	travelAPIClient TravelAPIClient
	l10nService     L10nService
}

func NewGRPCAviaOrderHandler(logger log.Logger, travelAPIClient TravelAPIClient, l10nService L10nService, config *avia.OrderConfig) *GRPCHandler {
	return &GRPCHandler{
		logger:          logger,
		config:          config,
		travelAPIClient: travelAPIClient,
		l10nService:     l10nService,
	}
}

func (h *GRPCHandler) GetL10NKeysetName() string {
	return l10nKeyset
}

func (h *GRPCHandler) GetL10NKeys() []string {
	return []string{
		string(defaultWaitingTitleKey),
		string(defaultWaitingMessageKey),
		string(preparingOrderTitleKey),
		string(preparingOrderMessageKey),
		string(hpButtonTextKey),
		string(hpResultCellPrimaryTextSuccessKey),
		string(hpResultCellPrimaryTextPaymentFailedKey),
		string(hpResultCellPrimaryTextBookingCancelledKey),
		string(hpResultCellPrimaryTextUnknownKey),
		string(hpResultCellSecondaryTextSuccessKey),
		string(hpResultCellSecondaryTextPaymentFailedKey),
		string(hpResultCellSecondaryTextBookingCancelledKey),
		string(hpResultCellSecondaryTextUnknownKey),
		string(hpResultCellWhatsappTextKey),
		string(hpResultCellCallDataTextKey),
		string(hpResultCellCallDataPhoneKey),
		string(hpSupportCellCallDataTextKey),
		string(hpSupportCellCallDataPhoneKey),
		string(hpSupportCellPrimaryTextKey),
		string(hpSupportCellSecondaryTextKey),
	}
}

func (h *GRPCHandler) getTranslations(ctx context.Context) (*l10n.Keyset, error) {
	lang := common.GetLanguage(ctx)
	keyset, err := h.l10nService.Get(l10nKeyset, lang)
	if err != nil {
		if errors.Is(err, l10n.UnsupportedLanguage) {
			return nil, status.Error(codes.InvalidArgument, fmt.Sprintf("unsupported language '%s'", lang))
		}
		msg := fmt.Sprintf("no keyset for %s, %s", l10nKeyset, lang)
		ctxlog.Error(ctx, h.logger, msg, log.Error(err))
		return nil, status.Error(codes.Unknown, msg)
	}
	return keyset, nil
}

func (h *GRPCHandler) getTranslation(keyset *l10n.Keyset, key translationKey) string {
	// Не проверяем нашелся или нет ключ,
	// потому что проверка ключей происходит на старте приложения и при обновлении данных из танкера.
	// Проверяется наличие ключей, которые мы возвращаем в GetL10NKeys()
	return keyset.Keys[string(key)]
}

//TODO(adurnev) delete?
func (h *GRPCHandler) GetOrderStatus(ctx context.Context, req *commonOrderAPI.GetOrderReq) (*commonOrderAPI.GetOrderRsp, error) {
	if !req.ForceFullOrderInfo {
		orderStatus, err := h.travelAPIClient.GetAviaOrderStatus(ctx, req.OrderId)
		if err != nil {
			ctxlog.Error(ctx, h.logger, "error get avia order status", log.Error(err))
			return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
		}
		currentStatus := convertDisplayOrderStatusToProto(orderStatus.DisplayOrderState)
		if _, ok := common.FinalStatuses[currentStatus]; !ok {
			purchaseToken, err := common.GetTokenFromURL(orderStatus.PaymentURL, purchaseTokenKey)
			if err != nil {
				ctxlog.Error(ctx, h.logger, "invalid get_order_status response", log.Error(err))
				return nil, status.Error(codes.Unknown, "invalid order info")
			}
			return &commonOrderAPI.GetOrderRsp{
				Status:        currentStatus,
				PurchaseToken: purchaseToken,
			}, nil
		}
	}
	hp, err := h.travelAPIClient.GetAviaHappyPage(ctx, req.OrderId)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}
	hpOrderstatus := convertDisplayOrderStatusToProto(hp.Order.EdisplayOrderState)

	keyset, err := h.getTranslations(ctx)
	if err != nil {
		return nil, err
	}
	getTranslation := func(key translationKey) string {
		return h.getTranslation(keyset, key)
	}
	return &commonOrderAPI.GetOrderRsp{
		Status:        hpOrderstatus,
		HappyPageData: h.buildHappyPage(hp, &hpOrderstatus, getTranslation),
	}, nil
}

func (h *GRPCHandler) GetExpectedAction(ctx context.Context, req *commonOrderAPI.GetExpectedActionReq) (*commonOrderAPI.GetExpectedActionRsp, error) {
	keyset, err := h.getTranslations(ctx)
	if err != nil {
		return nil, err
	}
	getTranslation := func(key translationKey) string {
		return h.getTranslation(keyset, key)
	}

	orderStatus, err := h.travelAPIClient.GetAviaOrderStatus(ctx, req.OrderId)
	if err != nil {
		ctxlog.Error(ctx, h.logger, "error get avia order status", log.Error(err))
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
	}

	// Init payment
	// https://a.yandex-team.ru/arc_vcs/travel/frontend/portal/src/projects/avia/pages/AviaBookingPayment/BookingPayment.tsx?rev=r9331608#L126
	// Если invoiceState отсутствует, то это сигнализирует о том
	// что оплата не была проинициализирована.
	// Оплату заказа можно начинать только после статуса OS_WAIT_CARD_TOKENIZED
	if orderStatus.OrderState == models.OrderStatusOsWaitCardTokenized && orderStatus.InvoiceState == nil {
		err = h.travelAPIClient.AviaStartPayment(ctx, req.OrderId)
		if err != nil {
			ctxlog.Error(ctx, h.logger, "error start avia payment", log.Error(err))
			return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
		}
		response := commonOrderAPI.GetExpectedActionRsp{}
		response.ExpectedAction = &commonOrderAPI.GetExpectedActionRsp_Waiting_{
			Waiting: &commonOrderAPI.GetExpectedActionRsp_Waiting{
				Title:    getTranslation(defaultWaitingTitleKey),
				Subtitle: getTranslation(defaultWaitingMessageKey),
				DelayMs:  200,
			},
		}
		return &response, nil
	}

	switch orderStatus.DisplayOrderState {
	case models.OrderStatusDisplayOsInProgress:
		response := commonOrderAPI.GetExpectedActionRsp{}
		response.ExpectedAction = &commonOrderAPI.GetExpectedActionRsp_Waiting_{
			Waiting: &commonOrderAPI.GetExpectedActionRsp_Waiting{
				Title:    getTranslation(defaultWaitingTitleKey),
				Subtitle: getTranslation(defaultWaitingMessageKey),
				DelayMs:  200,
			},
		}
		return &response, nil

	case models.OrderStatusDisplayOsAwaitsPayment:
		if orderStatus.ConfirmationURL != nil && len(*orderStatus.ConfirmationURL) > 0 {
			response := commonOrderAPI.GetExpectedActionRsp{}
			response.ExpectedAction = &commonOrderAPI.GetExpectedActionRsp_Webview{
				Webview: &commonOrderAPI.GetExpectedActionRsp_ShowWebview{
					Url:                *orderStatus.ConfirmationURL,
					MaxShowTimeSeconds: uint32((20 * time.Minute).Seconds()),
				},
			}
			return &response, nil
		} else if orderStatus.PaymentURL != nil && len(*orderStatus.PaymentURL) > 0 {
			purchaseToken, err := common.GetTokenFromURL(orderStatus.PaymentURL, purchaseTokenKey)
			if err != nil {
				ctxlog.Error(ctx, h.logger, "invalid get_order_status response", log.Error(err))
				return nil, status.Error(codes.Unknown, "invalid order info")
			}
			response := commonOrderAPI.GetExpectedActionRsp{}
			response.ExpectedAction = &commonOrderAPI.GetExpectedActionRsp_Payment_{
				Payment: &commonOrderAPI.GetExpectedActionRsp_Payment{
					Value: &commonOrderAPI.GetExpectedActionRsp_Payment_PurchaseToken{
						PurchaseToken: purchaseToken.GetValue(),
					},
				},
			}
			return &response, nil
		} else {
			response := commonOrderAPI.GetExpectedActionRsp{}
			response.ExpectedAction = &commonOrderAPI.GetExpectedActionRsp_Waiting_{
				Waiting: &commonOrderAPI.GetExpectedActionRsp_Waiting{
					Title:    getTranslation(preparingOrderTitleKey),
					Subtitle: getTranslation(preparingOrderMessageKey),
					DelayMs:  200,
				},
			}
			return &response, nil
		}

	case models.OrderStatusDisplayOsFulfilled,
		models.OrderStatusDisplayOsCancelled,
		models.OrderStatusDisplayOsRefunded:
		hp, err := h.travelAPIClient.GetAviaHappyPage(ctx, req.OrderId)
		if err != nil {
			return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, serviceName)
		}
		hpOrderStatus := convertOrderStatusToProto(hp.Order.EdisplayOrderState)

		response := commonOrderAPI.GetExpectedActionRsp{}
		response.ExpectedAction = &commonOrderAPI.GetExpectedActionRsp_HappyPage{
			HappyPage: h.buildHappyPage(hp, &hpOrderStatus, getTranslation),
		}
		return &response, nil
	}

	return nil, status.Error(codes.Unknown, serviceName+" unexpected error")
}

func (h *GRPCHandler) buildHappyPage(hp *models.GetAviaHappyPageRsp, status *commonOrderAPI.OrderStatus, getTransaction func(key translationKey) string) *commonOrderAPI.HappyPageData {
	order := hp.Order
	finalStatus := common.BuildFinalStatus(status)
	flights := buildFlights(order.AirReservation.Segments)
	stations := buildStations(order.Reference.Airports)
	return &commonOrderAPI.HappyPageData{
		Status:     finalStatus,
		ImageUrl:   h.buildImageURL(&finalStatus), //todo(adurnev) дозаполнить
		ButtonText: getTransaction(hpButtonTextKey),
		ResultCell: &commonOrderAPI.ResultCellData{
			PrimaryText:         buildPrimaryText(&finalStatus, getTransaction),
			SecondaryText:       buildSecondaryText(&finalStatus, getTransaction),
			Email:               order.Email,
			WhatsappSectionText: getTransaction(hpResultCellWhatsappTextKey),
			AviaBookingCode:     order.AirReservation.Pnr,
			AviaPaymentInfoData: &commonOrderAPI.AviaPaymentInfoData{
				Price:           order.Price.Value,
				ClassOfFlight:   aviaAPI.ServiceClass_SERVICE_CLASS_ECONOMY,
				PassengersCount: int32(len(order.Travellers)),
			},
			CallData: &commonOrderAPI.CallData{
				Phone: getTransaction(hpResultCellCallDataPhoneKey),
				Text:  getTransaction(hpResultCellCallDataTextKey),
			},
		},
		AviaBookedCell: &commonOrderAPI.AviaBookedCellData{
			AviaDetailsCell: []*commonOrderAPI.AviaDetailsCellData{
				{
					Snippet:   buildSnippet(hp, flights, stations),
					Companies: buildAviaCompany(order.Reference.Airlines),
				},
			},
			Flights:     flights,
			Settlements: buildSettlements(order.Reference.Settlements),
			Stations:    stations,
		},
		SupportCell: &commonOrderAPI.SupportCellData{
			CallData: &commonOrderAPI.CallData{
				Phone: getTransaction(hpSupportCellCallDataPhoneKey),
				Text:  getTransaction(hpSupportCellCallDataTextKey),
			},
			PrimaryText:   getTransaction(hpSupportCellPrimaryTextKey),
			SecondaryText: getTransaction(hpSupportCellSecondaryTextKey),
		},
	}
}

func (h *GRPCHandler) buildImageURL(status *commonOrderAPI.HappyPageFinalStatus) string {
	if status != nil && *status == commonOrderAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_SUCCESS {
		return h.config.HappyPageImageSuccessURL
	}
	return h.config.HappyPageImageFailedURL
}

func buildPrimaryText(status *commonOrderAPI.HappyPageFinalStatus, getTransaction func(key translationKey) string) string {
	switch *status {
	case commonOrderAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_SUCCESS:
		return getTransaction(hpResultCellPrimaryTextSuccessKey)
	case commonOrderAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_PAYMENT_FAILED:
		return getTransaction(hpResultCellPrimaryTextPaymentFailedKey)
	case commonOrderAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_BOOKING_CANCELLED:
		return getTransaction(hpResultCellPrimaryTextBookingCancelledKey)
	}
	return getTransaction(hpResultCellPrimaryTextUnknownKey)
}

func buildSecondaryText(status *commonOrderAPI.HappyPageFinalStatus, getTranslation func(key translationKey) string) string {
	switch *status {
	case commonOrderAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_SUCCESS:
		return getTranslation(hpResultCellSecondaryTextSuccessKey)
	case commonOrderAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_PAYMENT_FAILED:
		return getTranslation(hpResultCellSecondaryTextPaymentFailedKey)
	case commonOrderAPI.HappyPageFinalStatus_HAPPY_PAGE_FINAL_STATUS_BOOKING_CANCELLED:
		return getTranslation(hpResultCellSecondaryTextBookingCancelledKey)
	}
	return getTranslation(hpResultCellSecondaryTextUnknownKey)
}

//TODO(adurnev) может можно получить снипет из tdApi?
var layout = "2006-01-02T15:04:05-0700"

func buildSnippet(hp *models.GetAviaHappyPageRsp, flights map[string]*aviaAPI.SearchResultReference_Flight, stations map[uint64]*aviaAPI.SearchResultReference_Station) *aviaAPI.Snippet {
	//order := hp.Order
	forward := make([]string, 0)
	backward := make([]string, 0)
	if len(hp.Order.AirReservation.Segments) > 0 {
		segment := hp.Order.AirReservation.Segments[0]
		for _, flight := range segment.Flights {
			forward = append(forward, flight.ID)
		}
	}
	if len(hp.Order.AirReservation.Segments) > 1 {
		segment := hp.Order.AirReservation.Segments[1]
		for _, flight := range segment.Flights {
			backward = append(backward, flight.ID)
		}
	}

	flightReference := make(map[string]aviatdapiclient.Flight)
	for k, v := range flights {
		flightReference[k] = aviatdapiclient.Flight{
			Key:    k,
			Number: v.Number,
			Departure: aviatdapiclient.DateTime{
				Local: v.Departure, //TODO(adurnev) коррекно заполнить
			},
			Arrival: aviatdapiclient.DateTime{
				Local: v.Arrival,
			},
			AviaCompanyID: ptr.Uint64(v.AviaCompanyId),
			StationFromID: v.StationFromId,
			StationToID:   v.StationToId,
		}
	}

	stationReference := make(map[uint64]aviatdapiclient.Station)
	for k, v := range stations {
		stationReference[k] = aviatdapiclient.Station{
			ID:           k,
			Title:        v.Title,
			AviaCode:     v.AviaCode,
			SettlementID: &v.SettlementId,
		}
	}

	transfers, _ := search.BuildTransfers(forward, backward, false, flightReference, stationReference) //TODO(adurnev) selfConnect
	/*if err != nil {
		return nil, xerrors.Errorf("build snippet transfers error: %w", err)
	}*/
	forwardDurationMinutes, _ := search.CalcDurationMinutes(forward, flightReference)
	/*if err != nil {
		return nil, xerrors.Errorf("calc forward duration error: %w", err)
	}*/
	backwardDurationMinutes, _ := search.CalcDurationMinutes(backward, flightReference)
	/*if err != nil {
		return nil, xerrors.Errorf("calc backward duration error: %w", err)
	}*/

	result := &aviaAPI.Snippet{
		Forward:  forward,
		Backward: backward,
		Variant: &aviaAPI.Snippet_Variant{
			Price: &commonAPI.Price{
				Currency: hp.Order.Price.Currency,
				Value:    hp.Order.Price.Value,
			},
			Baggage: &aviaAPI.Snippet_Baggage{}, //TODO(adurnev) дозаполнить
			CarryOn: &aviaAPI.Snippet_CarryOn{},
		},
		//Badges:                  []*aviaAPI.Snippet_Badge{},
		Transfers:               transfers,
		ForwardDurationMinutes:  forwardDurationMinutes,
		BackwardDurationMinutes: backwardDurationMinutes,
	}
	return result
}

func buildStations(airports []models.Airport) map[uint64]*aviaAPI.SearchResultReference_Station {
	result := make(map[uint64]*aviaAPI.SearchResultReference_Station)

	for _, airport := range airports {
		id := uint64(airport.ID)
		result[id] = &aviaAPI.SearchResultReference_Station{
			Title:        airport.Title,
			Id:           id,
			AviaCode:     airport.Code,
			SettlementId: uint64(airport.Settlement),
			//TODO(adurnev) дозаполнить
		}
	}

	return result
}

func buildAviaCompany(airlines []models.Airline) []*commonOrderAPI.AviaCompanySnippetData {
	result := make([]*commonOrderAPI.AviaCompanySnippetData, len(airlines))

	for i, airline := range airlines {
		result[i] = &commonOrderAPI.AviaCompanySnippetData{
			Title:   airline.Title,
			LogoUrl: airline.LogoSvg,
			Id:      uint64(airline.ID),
		}
	}

	return result
}

func buildSettlements(settlements []models.Settlement) map[uint64]*aviaAPI.SearchResultReference_Settlement {
	result := make(map[uint64]*aviaAPI.SearchResultReference_Settlement)

	for _, settlement := range settlements {
		id := uint64(settlement.ID)
		result[id] = &aviaAPI.SearchResultReference_Settlement{
			Id:    id,
			Title: settlement.Title,
			//TODO(adurnev) дозаполнить
		}
	}

	return result
}

func buildFlights(segments []models.Segment) map[string]*aviaAPI.SearchResultReference_Flight {
	result := make(map[string]*aviaAPI.SearchResultReference_Flight)

	for _, segment := range segments {
		for _, flight := range segment.Flights {
			result[flight.ID] = &aviaAPI.SearchResultReference_Flight{
				Key:           flight.ID,
				Arrival:       flight.Arrival,
				Departure:     flight.Departure,
				Number:        flight.Number,
				AviaCompanyId: uint64(flight.OperatingAviaCompany),
				StationFromId: uint64(flight.From),
				StationToId:   uint64(flight.To),
				//TODO(adurnev) дозаполнить
				//DurationMinutes:
				//DateChanged:
			}
		}
	}

	return result
}

func (h *GRPCHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		aviaOrderAPI.RegisterAviaOrderAPIServer(server, h)
	}
}

//TODO(adurnev) пока не удаляю
func convertOrderStatusToProto(s models.OrderStatus) commonOrderAPI.OrderStatus {
	switch s {
	case models.OrderStatusOsAutoConfirmingInvoice, models.OrderStatusOsAutoRestoringInvoice, models.OrderStatusOsCheckAvailability, models.OrderStatusOsWaitOrderCreated, models.OrderStatusOsConfirmingOrder, models.OrderStatusOsConfirmingOrderItem, models.OrderStatusOsNew, models.OrderStatusOsCancellingInvoice, models.OrderStatusOsCancellingOrderItem:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_IN_PROGRESS

	case models.OrderStatusOsWaitCardTokenized:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_WAIT_START_PAYMENT

	case models.OrderStatusOsWaitOrderPaid:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_WAITING_FOR_PAYMENT

	case models.OrderStatusOsConfirmed:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_SUCCESS

	case models.OrderStatusOsCancelled, models.OrderStatusOsExternallyCancelled:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_CANCELLED
	}
	return commonOrderAPI.OrderStatus_ORDER_STATUS_UNKNOWN
}

func convertDisplayOrderStatusToProto(s models.OrderStatus) commonOrderAPI.OrderStatus {
	switch s {
	case models.OrderStatusDisplayOsInProgress:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_IN_PROGRESS

	case models.OrderStatusDisplayOsAwaitsPayment:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_WAIT_START_PAYMENT

	case models.OrderStatusDisplayOsFulfilled:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_SUCCESS

	case models.OrderStatusDisplayOsCancelled, models.OrderStatusDisplayOsRefunded:
		return commonOrderAPI.OrderStatus_ORDER_STATUS_CANCELLED
	}
	return commonOrderAPI.OrderStatus_ORDER_STATUS_UNKNOWN
}

type TravelAPIClient interface {
	GetAviaOrderStatus(ctx context.Context, orderID string) (*models.GetAviaOrderStatusRsp, error)
	GetAviaOrder(ctx context.Context, orderID string) (*models.GetAviaOrderRsp, error)
	GetAviaHappyPage(ctx context.Context, orderID string) (*models.GetAviaHappyPageRsp, error)
	AviaStartPayment(ctx context.Context, orderID string) error
}

type L10nService interface {
	Get(keysetName string, language string) (*l10n.Keyset, error)
}
