package handler

import (
	"context"
	"time"

	"golang.org/x/exp/slices"
	"google.golang.org/genproto/googleapis/type/date"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	priceAPI "a.yandex-team.ru/travel/app/backend/api/aviaprice/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia"
	"a.yandex-team.ru/travel/app/backend/internal/lib/clientscommon"
	"a.yandex-team.ru/travel/app/backend/internal/lib/priceindexclient"
	"a.yandex-team.ru/travel/app/backend/internal/points"
	"a.yandex-team.ru/travel/library/go/geobase/consts"
)

const serviceName = "avia price"
const formatDate = "2006-01-02T15:04:05"

type GRPCHandler struct {
	config           *avia.PriceConfig
	logger           log.Logger
	priceIndexClient priceindexclient.Client
	pointParser      points.IParser
}

func NewGRPCHandler(logger log.Logger, config *avia.PriceConfig, priceIndexClient priceindexclient.Client,
	pointParser points.IParser) *GRPCHandler {
	return &GRPCHandler{
		logger:           logger,
		config:           config,
		priceIndexClient: priceIndexClient,
		pointParser:      pointParser,
	}
}

func (h *GRPCHandler) GetPrice(ctx context.Context, req *priceAPI.PriceReq) (*priceAPI.PriceRsp, error) {
	result := priceAPI.PriceRsp{
		Data: make(map[string]*priceAPI.PriceItem, 0),
	}

	from, err := h.pointKeyToSettlementID(req.PointKeyFrom)
	if err != nil {
		return nil, err
	}
	to, err := h.pointKeyToSettlementID(req.PointKeyTo)
	if err != nil {
		return nil, err
	}
	if from == nil || to == nil {
		return &result, nil
	}

	// Делаем один запрос с множеством дат в диапозоне + 120 дней
	// https://a.yandex-team.ru/arc_vcs/travel/frontend/portal/server/services/AviaPriceIndexService/utilities/getCalendarPriceRequest/utilities/getCalendarPricesByDeparture.ts?rev=r9331608#L19
	dateStart := time.Now()
	if req.GetDateForward() != nil {
		df := *req.GetDateForward()
		dateStart = time.Date(int(df.Year), time.Month(df.Month), int(df.Day), 0, 0, 0, 0, time.UTC)
	}
	dateFinish := dateStart.AddDate(0, 0, h.config.CountDays)

	dfFix := &date.Date{
		Year:  int32(dateStart.Year()),
		Month: int32(dateStart.Month()),
		Day:   int32(dateStart.Day()),
	}
	requests := make([]*priceindexclient.BatchReq, 0)
	for dateStart.Before(dateFinish) {
		var df *date.Date
		var db *date.Date

		dateCurrent := &date.Date{
			Year:  int32(dateStart.Year()),
			Month: int32(dateStart.Month()),
			Day:   int32(dateStart.Day()),
		}
		if req.GetDateForward() == nil {
			df = dateCurrent
			db = nil
		} else {
			df = dfFix
			db = dateCurrent
		}

		r := priceindexclient.BatchReq{
			FromID:       *from,
			ToID:         *to,
			DateForward:  df,
			DateBackward: db,
		}
		if req.Passengers != nil {
			a := int(req.Passengers.Adults)
			r.Adults = &a
			c := int(req.Passengers.Children)
			r.Children = &c
			i := int(req.Passengers.Infants)
			r.Infants = &i
		}

		requests = append(requests, &r)
		dateStart = dateStart.Add(consts.Day)
	}
	request := priceindexclient.PriceBatchReq{
		Requests: requests,
	}
	response, err := h.priceIndexClient.GetPriceBatch(ctx, &request)
	if err != nil {
		return nil, clientscommon.ConvertHTTPErrorToGRPCError(ctx, h.logger, err, "price index")
	}

	median := h.getMedian(response.Data) * h.config.MedianCoefficient
	now := time.Now()
	m := result.Data
	for _, data := range response.Data {
		var d string
		if data.BackwardDate != nil {
			d = *data.BackwardDate
		} else {
			d = data.ForwardDate
		}
		v := float64(data.MinPrice.Value)
		m[d] = &priceAPI.PriceItem{
			Price: &commonAPI.Price{
				Value:    v,
				Currency: *clientscommon.GetCorrectedCurrency(&data.MinPrice.Currency),
			},
			IsLow:     v < median,
			IsRoughly: !h.isRecentPrice(data.UpdatedAt, now),
		}
	}
	return &result, nil
}

// https://a.yandex-team.ru/arcadia/travel/frontend/portal/server/services/AviaPriceIndexService/utilities/convertCalendarPrices.ts?rev=r9331608#L46
func (h *GRPCHandler) isRecentPrice(updatedAtStr *string, now time.Time) bool {
	if updatedAtStr == nil {
		return false
	}
	updatedAt, err := time.Parse(formatDate, *updatedAtStr)
	if err != nil {
		return false
	}

	return now.Sub(updatedAt) <= h.config.PriceExpirationInterval
}

// https://a.yandex-team.ru/arcadia/travel/frontend/portal/server/services/AviaPriceIndexService/utilities/convertCalendarPrices.ts?rev=r9331608#L18
func (h *GRPCHandler) getMedian(batchData []priceindexclient.PriceBatchData) float64 {
	if len(batchData) == 0 {
		return 0
	}
	prices := make([]int, 0)
	for _, data := range batchData {
		if data.MinPrice.Value > 0 {
			prices = append(prices, data.MinPrice.Value)
		}
	}
	slices.Sort(prices)
	return float64(prices[len(prices)/2])
}

func (h *GRPCHandler) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		priceAPI.RegisterAviaPriceAPIServer(server, h)
	}
}

func (h *GRPCHandler) pointKeyToSettlementID(pointKey string) (*int, error) {
	p, err := h.pointParser.ParseByPointKey(pointKey)
	if err != nil {
		if xerrors.Is(err, points.NotExistsErr) {
			return nil, status.Error(codes.NotFound, err.Error())
		}
		if xerrors.Is(err, points.UnexpectedFormatErr) {
			return nil, status.Error(codes.InvalidArgument, err.Error())
		}
		return nil, err
	}
	id := p.SettlementID()
	if id == nil {
		return nil, nil
	}
	intID := int(*id)
	return &intID, nil
}
