package badges

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	aviaApi "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/library/go/containers"
)

type BadgeSectionConfig struct {
	Priority    []aviaApi.Snippet_BadgeType
	MaxElements int
}

type CombinedBadgeObserverBuilderConfig struct {
	Sections            []BadgeSectionConfig
	Popular             PopularBadgeObserverBuilderConfig
	Comfortable         ComfortableBadgeObserverBuilderConfig
	BadgesOnlyForSort   map[aviaApi.Snippet_BadgeType][]aviaApi.SearchSort
	BadgesDeleteOnExist map[aviaApi.Snippet_BadgeType][]aviaApi.Snippet_BadgeType
}

var DefaultCombinedBadgeObserverBuilderConfig = CombinedBadgeObserverBuilderConfig{
	Sections: []BadgeSectionConfig{
		{
			Priority: []aviaApi.Snippet_BadgeType{
				aviaApi.Snippet_BADGE_TYPE_BEST_PRICE,
			},
			MaxElements: 1,
		},
		{
			Priority: []aviaApi.Snippet_BadgeType{
				aviaApi.Snippet_BADGE_TYPE_BOOK_ON_YANDEX,
				aviaApi.Snippet_BADGE_TYPE_AVIACOMPANY_DIRECT_SELLING,
				aviaApi.Snippet_BADGE_TYPE_PLUS_CASHBACK,
				aviaApi.Snippet_BADGE_TYPE_CHARTER,
				aviaApi.Snippet_BADGE_TYPE_SPECIAL_CONDITIONS,
				aviaApi.Snippet_BADGE_TYPE_COMFY,
				aviaApi.Snippet_BADGE_TYPE_POPULAR,
			},
			MaxElements: 1,
		},
	},
	Popular:     DefaultPopularBadgeObserverBuilderConfig,
	Comfortable: DefaultComfortableBadgeObserverBuilderConfig,
	BadgesOnlyForSort: map[aviaApi.Snippet_BadgeType][]aviaApi.SearchSort{
		aviaApi.Snippet_BADGE_TYPE_CHARTER:            {aviaApi.SearchSort_SEARCH_SORT_RECOMMENDED_FIRST},
		aviaApi.Snippet_BADGE_TYPE_SPECIAL_CONDITIONS: {aviaApi.SearchSort_SEARCH_SORT_RECOMMENDED_FIRST},
		aviaApi.Snippet_BADGE_TYPE_COMFY:              {aviaApi.SearchSort_SEARCH_SORT_RECOMMENDED_FIRST},
		aviaApi.Snippet_BADGE_TYPE_POPULAR:            {aviaApi.SearchSort_SEARCH_SORT_RECOMMENDED_FIRST},
	},
	BadgesDeleteOnExist: map[aviaApi.Snippet_BadgeType][]aviaApi.Snippet_BadgeType{
		aviaApi.Snippet_BADGE_TYPE_AVIACOMPANY_DIRECT_SELLING: {aviaApi.Snippet_BADGE_TYPE_BOOK_ON_YANDEX},
	},
}

type CombinedBadgeObserverBuilder struct {
	cfg      *CombinedBadgeObserverBuilderConfig
	builders []BadgeObserverBuilder
}

func NewCombinedBadgeObserverBuilder(cfg *CombinedBadgeObserverBuilderConfig, logger log.Logger) *CombinedBadgeObserverBuilder {
	return &CombinedBadgeObserverBuilder{
		builders: []BadgeObserverBuilder{
			NewBoyBadgeObserverBuilder(logger),
			NewCharterBadgeObserverBuilder(logger),
			NewComfortableBadgeObserverBuilder(&cfg.Comfortable, logger),
			NewFromCompanyBadgeObserverBuilder(logger),
			NewPopularBadgeObserverBuilder(&cfg.Popular, logger),
			NewBestPriceBadgeObserverBuilder(logger),
		},
		cfg: cfg,
	}
}

func (b *CombinedBadgeObserverBuilder) BuildToCache(
	ctx context.Context,
	clients *AviaClients,
	qid *searchcontext.QID,
	reference *aviatdapiclient.SearchResultReference,
) ToCacheBadgeObserver {
	var workers []ToCacheBadgeObserver
	for _, b := range b.builders {
		workers = append(workers, b.BuildToCache(ctx, clients, qid, reference))
	}
	return &ToCacheCombinedBadgeObserver{
		workers: workers,
	}
}

func (b *CombinedBadgeObserverBuilder) BuildToResponse(
	ctx context.Context,
	clients *AviaClients,
	qid *searchcontext.QID,
	searchSort aviaApi.SearchSort,
	reference *aviaSearchProto.Reference,
	cacheSnippetStats *aviaSearchProto.CacheSnippetStats,
) ToResponseBadgeObserver {
	var workers []ToResponseBadgeObserver
	for _, b := range b.builders {
		workers = append(workers, b.BuildToResponse(ctx, clients, qid, searchSort, reference, cacheSnippetStats))
	}
	return &ToResponseCombinedBadgeObserver{
		cfg:        b.cfg,
		searchSort: searchSort,
		workers:    workers,
	}
}

type ToCacheCombinedBadgeObserver struct {
	workers []ToCacheBadgeObserver
}

func (pc *ToCacheCombinedBadgeObserver) ObserveAllSnippets(snippets map[string]*aviaSearchProto.Snippet, cacheSnippetStats *aviaSearchProto.CacheSnippetStats) (map[string]*aviaSearchProto.Snippet, *aviaSearchProto.CacheSnippetStats) {
	for _, w := range pc.workers {
		snippets, cacheSnippetStats = w.ObserveAllSnippets(snippets, cacheSnippetStats)
	}
	return snippets, cacheSnippetStats
}

func (pc *ToCacheCombinedBadgeObserver) ObserveRawVariant(variant *aviatdapiclient.SearchResultPrice, badges []*aviaSearchProto.Badge) []*aviaSearchProto.Badge {
	for _, w := range pc.workers {
		badges = w.ObserveRawVariant(variant, badges)
	}
	return badges
}

func (pc *ToCacheCombinedBadgeObserver) ObserveSnippet(fare *aviatdapiclient.SearchResultFare, snippet *aviaSearchProto.Snippet) *aviaSearchProto.Snippet {
	for _, w := range pc.workers {
		snippet = w.ObserveSnippet(fare, snippet)
	}
	return snippet
}

type ToResponseCombinedBadgeObserver struct {
	cfg        *CombinedBadgeObserverBuilderConfig
	searchSort aviaApi.SearchSort
	workers    []ToResponseBadgeObserver
}

func (t *ToResponseCombinedBadgeObserver) removeBadgesIfRelativeExists(badges []*aviaApi.Snippet_Badge) []*aviaApi.Snippet_Badge {
	badgesOnSnippet := containers.SetOf[aviaApi.Snippet_BadgeType]()
	for _, badge := range badges {
		badgesOnSnippet.Add(badge.Type)
	}

	newBadges := make([]*aviaApi.Snippet_Badge, 0)

	for _, badge := range badges {
		relatives, exists := t.cfg.BadgesDeleteOnExist[badge.Type]
		if !exists {
			newBadges = append(newBadges, badge)
			continue
		}

		relativeExists := false

		for _, relativeBadge := range relatives {
			if badgesOnSnippet.Contains(relativeBadge) {
				relativeExists = true
				break
			}
		}

		if !relativeExists {
			newBadges = append(newBadges, badge)
		}
	}

	return newBadges
}

func (t *ToResponseCombinedBadgeObserver) sortAndSliceBadgesInSection(
	badges []*aviaApi.Snippet_Badge,
	section *BadgeSectionConfig,
) []*aviaApi.Snippet_Badge {
	sortedBadges := make([]*aviaApi.Snippet_Badge, 0, section.MaxElements)

	for _, sortedBadge := range section.Priority {
		for _, badge := range badges {
			if len(sortedBadges) >= section.MaxElements {
				return sortedBadges
			}
			if sortedBadge == badge.Type {
				sortedBadges = append(sortedBadges, badge)
			}
		}
	}

	return sortedBadges
}

func (t *ToResponseCombinedBadgeObserver) sortAndSliceBadges(badges []*aviaApi.Snippet_Badge) []*aviaApi.Snippet_Badge {
	sortedBadges := make([]*aviaApi.Snippet_Badge, 0)

	for _, section := range t.cfg.Sections {
		newBadges := t.sortAndSliceBadgesInSection(badges, &section)
		sortedBadges = append(sortedBadges, newBadges...)
	}

	return sortedBadges
}

func (t *ToResponseCombinedBadgeObserver) filterBadgesBySnippetSort(badges []*aviaApi.Snippet_Badge) []*aviaApi.Snippet_Badge {
	result := make([]*aviaApi.Snippet_Badge, 0)

	for _, badge := range badges {
		allowedSorts, exists := t.cfg.BadgesOnlyForSort[badge.Type]
		if !exists {
			result = append(result, badge)
			continue
		}

		for _, allowedSort := range allowedSorts {
			if allowedSort == t.searchSort {
				result = append(result, badge)
				break
			}
		}
	}

	return result
}

func (t *ToResponseCombinedBadgeObserver) ObserveSnippet(snippet *aviaApi.Snippet, stats *SnippetStats) *aviaApi.Snippet {
	for _, w := range t.workers {
		snippet = w.ObserveSnippet(snippet, stats)
	}

	snippet.Badges = t.removeBadgesIfRelativeExists(snippet.Badges)
	snippet.Badges = t.filterBadgesBySnippetSort(snippet.Badges)
	snippet.Badges = t.sortAndSliceBadges(snippet.Badges)

	return snippet
}
