package badges

import (
	"context"
	"math"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	mathUtils "a.yandex-team.ru/library/go/x/math"
	aviaApi "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/library/go/containers"
)

// Realisation from https://wiki.yandex-team.ru/avia/search-results/
// PLEASE EDIT THE CONTENT UNDER LINK IF YOU ARE GOING TO CHANGE THIS FILE

// Flight helpers

func (o *ToCacheComfortableBadgeObserver) getFlightsWithDepartureAndArrivalTime() containers.Set[string] {
	result := containers.SetOf[string]()

	for flightKey, flight := range o.flights {
		if flight.Departure.Local != "" && flight.Arrival.Local != "" {
			result.Add(flightKey)
		}
	}

	return result
}

func (o *ToCacheComfortableBadgeObserver) getFlightsWithDepartureTimeInInterval() containers.Set[string] {
	result := containers.SetOf[string]()

	for flightKey, flight := range o.flights {
		dep, err := time.Parse("2006-01-02T15:04:05", flight.Departure.Local)
		if err != nil {
			continue
		}

		minutes := dep.Hour()*60 + dep.Minute()
		if o.cfg.MinDepartureTimeInMinutes < minutes && minutes < o.cfg.MaxDepartureTimeInMinutes {
			result.Add(flightKey)
		}
	}

	return result
}

// FLight & snippet helpers

func (o *ToCacheComfortableBadgeObserver) checkIfSnippetHasAllFlightsFromSet(snippet *aviaSearchProto.Snippet, flights containers.Set[string]) bool {
	for _, way := range snippet.Forward {
		if !flights.Contains(way) {
			return false
		}
	}

	for _, way := range snippet.Backward {
		if !flights.Contains(way) {
			return false
		}
	}

	return true
}

func (o *ToCacheComfortableBadgeObserver) filterSnippetsByHasAllFlightsFromSet(snippetKeys []string, flights containers.Set[string]) []string {
	result := make([]string, 0)

	for _, snippetKey := range snippetKeys {
		snippet := o.snippets[snippetKey]
		if o.checkIfSnippetHasAllFlightsFromSet(snippet, flights) {
			result = append(result, snippetKey)
		}
	}

	return result
}

// Snippets with minimum changes

func GetMaxChangesInVariant(snippet *aviaSearchProto.Snippet) int {
	if len(snippet.Backward) > 0 {
		return len(snippet.Forward) - 1 + len(snippet.Backward) - 1
	} else {
		return len(snippet.Forward) - 1
	}

}

func GetMinChangesInVariants(snippets map[string]*aviaSearchProto.Snippet) int {
	minChangesInVariant := 9

	for _, snippet := range snippets {
		minChangesInVariant = mathUtils.MinInt(minChangesInVariant, GetMaxChangesInVariant(snippet))
		if minChangesInVariant == 0 {
			return 0
		}
	}

	return minChangesInVariant
}

func (o *ToCacheComfortableBadgeObserver) FilterMinChanges(snippetKeys []string) []string {
	minChanges := GetMinChangesInVariants(o.snippets)

	result := make([]string, 0)
	for _, snippetKey := range snippetKeys {
		if minChanges == GetMaxChangesInVariant(o.snippets[snippetKey]) {
			result = append(result, snippetKey)
		}
	}

	return result
}

// Snippets with no airport change

func (o *ToCacheComfortableBadgeObserver) checkIfWayHasNoAirportChange(flightKeys []string) bool {
	if len(flightKeys) < 2 {
		return true
	}

	for i := 0; i < len(flightKeys)-1; i++ {
		if o.flights[flightKeys[i]].StationToID != o.flights[flightKeys[i+1]].StationFromID {
			return false
		}
	}

	return true
}

func (o *ToCacheComfortableBadgeObserver) filterSnippetsByNoAirportChange(snippetKeys []string) []string {
	result := make([]string, 0)
	for _, snippetKey := range snippetKeys {
		snippet := o.snippets[snippetKey]
		if o.checkIfWayHasNoAirportChange(snippet.Forward) && (len(snippet.Backward) == 0 ||
			o.checkIfWayHasNoAirportChange(snippet.Backward)) {
			result = append(result, snippetKey)
		}
	}

	return result
}

// Snippets with near minimum total time

func (o *ToCacheComfortableBadgeObserver) FilterByNearMinTime(snippetKeys []string) []string {
	minTime := 10000000

	for _, snippet := range o.snippets {
		minTime = mathUtils.MinInt(minTime, int(snippet.ForwardDurationMinutes+snippet.BackwardDurationMinutes))
	}

	result := make([]string, 0)
	for _, snippetKey := range snippetKeys {
		snippet := o.snippets[snippetKey]
		snippetTime := float64(int(snippet.ForwardDurationMinutes + snippet.BackwardDurationMinutes))
		maxFlightDurationDelta := math.Max(float64(o.cfg.MaxFlightTimeDiffInMinutes), snippetTime*o.cfg.MaxFlightTimeDiffInPercents)

		if snippetTime-float64(minTime) <= maxFlightDurationDelta {
			result = append(result, snippetKey)
		}
	}

	return result
}

type ComfortableBadgeObserverBuilderConfig struct {
	MaxFlightTimeDiffInMinutes  int
	MaxFlightTimeDiffInPercents float64
	MinDepartureTimeInMinutes   int
	MaxDepartureTimeInMinutes   int
}

var DefaultComfortableBadgeObserverBuilderConfig = ComfortableBadgeObserverBuilderConfig{
	MaxFlightTimeDiffInMinutes:  15,
	MaxFlightTimeDiffInPercents: 0.2,
	MinDepartureTimeInMinutes:   6 * 60,
	MaxDepartureTimeInMinutes:   23 * 60,
}

type ComfortableBadgeObserverBuilder struct {
	BaseBadgeObserverBuilder
	logger log.Logger
	cfg    *ComfortableBadgeObserverBuilderConfig
}

func NewComfortableBadgeObserverBuilder(cfg *ComfortableBadgeObserverBuilderConfig, logger log.Logger) *ComfortableBadgeObserverBuilder {
	return &ComfortableBadgeObserverBuilder{
		logger: logger,
		cfg:    cfg,
	}
}

func (b *ComfortableBadgeObserverBuilder) BuildToCache(
	_ context.Context,
	_ *AviaClients,
	_ *searchcontext.QID,
	reference *aviatdapiclient.SearchResultReference,
) ToCacheBadgeObserver {
	flightsMap := make(map[string]*aviatdapiclient.Flight)
	for i, flight := range reference.Flights {
		flightsMap[flight.Key] = &reference.Flights[i]
	}

	return &ToCacheComfortableBadgeObserver{
		flights: flightsMap,
		cfg:     b.cfg,
	}
}

func (b *ComfortableBadgeObserverBuilder) BuildToResponse(
	_ context.Context,
	_ *AviaClients,
	_ *searchcontext.QID,
	_ aviaApi.SearchSort,
	_ *aviaSearchProto.Reference,
	cacheSnippetStats *aviaSearchProto.CacheSnippetStats,
) ToResponseBadgeObserver {
	if cacheSnippetStats == nil {
		return &ToResponseComfortableBadgeObserver{}
	} else {
		return &ToResponseComfortableBadgeObserver{
			snippetsForBadge: containers.SetOf[string](cacheSnippetStats.SnippetKeysForComfortableBadge...),
		}
	}
}

type ToCacheComfortableBadgeObserver struct {
	ToCacheBaseBadgeObserver
	flights  map[string]*aviatdapiclient.Flight
	snippets map[string]*aviaSearchProto.Snippet
	cfg      *ComfortableBadgeObserverBuilderConfig
}

func (o *ToCacheComfortableBadgeObserver) ObserveAllSnippets(snippets map[string]*aviaSearchProto.Snippet, cacheSnippetStats *aviaSearchProto.CacheSnippetStats) (map[string]*aviaSearchProto.Snippet, *aviaSearchProto.CacheSnippetStats) {
	o.snippets = snippets

	snippetKeys := make([]string, 0, len(snippets))
	for snippetKey := range snippets {
		snippetKeys = append(snippetKeys, snippetKey)
	}

	snippetKeys = o.filterSnippetsByHasAllFlightsFromSet(snippetKeys, o.getFlightsWithDepartureAndArrivalTime())
	snippetKeys = o.FilterMinChanges(snippetKeys)

	if newKeys := o.filterSnippetsByNoAirportChange(snippetKeys); len(newKeys) != 0 {
		snippetKeys = newKeys
	}

	snippetKeys = o.FilterByNearMinTime(snippetKeys)

	if newKeys := o.filterSnippetsByHasAllFlightsFromSet(snippetKeys, o.getFlightsWithDepartureTimeInInterval()); len(newKeys) > 0 {
		snippetKeys = newKeys
	}

	cacheSnippetStats.SnippetKeysForComfortableBadge = snippetKeys
	return snippets, cacheSnippetStats
}

type ToResponseComfortableBadgeObserver struct {
	snippetsForBadge containers.Set[string]
}

func (t *ToResponseComfortableBadgeObserver) ObserveSnippet(snippet *aviaApi.Snippet, stats *SnippetStats) *aviaApi.Snippet {
	if searchcommon.CheckOnlyPriceEqual(snippet.Variant.Price, stats.BestPrice) && t.snippetsForBadge.Contains(snippet.Key) {
		snippet.Badges = append(snippet.Badges, &aviaApi.Snippet_Badge{
			Type: aviaApi.Snippet_BADGE_TYPE_COMFY,
		})
	}
	return snippet
}
