package badges

import (
	"context"
	"strings"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviabackendclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/library/go/containers"
)

type PopularBadgeObserverBuilderConfig struct {
	FetchTopFlightsLimit int
}

var DefaultPopularBadgeObserverBuilderConfig = PopularBadgeObserverBuilderConfig{
	FetchTopFlightsLimit: 100,
}

type PopularBadgeObserverBuilder struct {
	BaseBadgeObserverBuilder
	cfg    *PopularBadgeObserverBuilderConfig
	logger log.Logger
}

func NewPopularBadgeObserverBuilder(cfg *PopularBadgeObserverBuilderConfig, logger log.Logger) *PopularBadgeObserverBuilder {
	return &PopularBadgeObserverBuilder{
		cfg:    cfg,
		logger: logger,
	}
}

func (b *PopularBadgeObserverBuilder) getTopFlightsByQid(ctx context.Context, aviaBackendClient BackendClient, qid *searchcontext.QID, useForward bool) (*aviabackendclient.TopFlightsRsp, error) {
	var date string
	if useForward {
		date = qid.QKey.DateForward.Format(timeformats.RFC3339Date)
	} else {
		date = qid.QKey.DateBackward.Format(timeformats.RFC3339Date)
	}
	return aviaBackendClient.TopFlights(
		ctx,
		qid.QKey.NationalVersion,
		qid.Lang,
		qid.QKey.PointFromKey,
		qid.QKey.PointToKey,
		date,
		b.cfg.FetchTopFlightsLimit,
	)
}

func (b *PopularBadgeObserverBuilder) BuildToCache(
	ctx context.Context,
	clients *AviaClients,
	qid *searchcontext.QID,
	reference *aviatdapiclient.SearchResultReference,
) ToCacheBadgeObserver {
	topFlightsForwardRsp, err := b.getTopFlightsByQid(ctx, clients.AviaBackendClient, qid, true)
	if err != nil {
		ctxlog.Error(ctx, b.logger, "Error while checking for popular", log.Error(err))
	}
	topFlightsForward := collectNumbers(topFlightsForwardRsp)

	topFlightsBackward := containers.Set[string]{}
	if !qid.QKey.DateBackward.IsZero() {
		topFlightsBackwardRsp, err := b.getTopFlightsByQid(ctx, clients.AviaBackendClient, qid, false)
		if err != nil {
			ctxlog.Error(ctx, b.logger, "Error while checking for popular", log.Error(err))
		}
		topFlightsBackward = collectNumbers(topFlightsBackwardRsp)
	}

	return &ToCachePopularBadgeObserver{
		logger:             b.logger,
		topFlightsForward:  topFlightsForward,
		topFlightsBackward: topFlightsBackward,
	}
}

type ToCachePopularBadgeObserver struct {
	ToCacheBaseBadgeObserver
	logger             log.Logger
	topFlightsForward  containers.Set[string]
	topFlightsBackward containers.Set[string]
}

func (pc *ToCachePopularBadgeObserver) checkFlightsForPopular(topFlights containers.Set[string], flightNumbers []string) bool {
	joinedNumbers := strings.Join(flightNumbers, "-")

	return topFlights.Contains(joinedNumbers)
}

func (pc *ToCachePopularBadgeObserver) CheckIfSnippetIsPopular(snippet *aviaSearchProto.Snippet) bool {
	if pc.topFlightsForward == nil {
		return false
	}

	isPopular := pc.checkFlightsForPopular(pc.topFlightsForward, snippet.Forward)
	if len(snippet.Backward) == 0 {
		return isPopular
	}

	if !isPopular || pc.topFlightsBackward == nil {
		return false
	}
	return pc.checkFlightsForPopular(pc.topFlightsBackward, snippet.Backward)
}

func collectNumbers(topFlights *aviabackendclient.TopFlightsRsp) containers.Set[string] {
	numbers := containers.Set[string]{}
	if topFlights == nil {
		return numbers
	}

	for _, group := range topFlights.Data {
		for _, elem := range group {
			numbers.Add(elem.Numbers)
		}
	}
	return numbers
}

func (pc *ToCachePopularBadgeObserver) ObserveAllSnippets(snippets map[string]*aviaSearchProto.Snippet, cacheSnippetStats *aviaSearchProto.CacheSnippetStats) (map[string]*aviaSearchProto.Snippet, *aviaSearchProto.CacheSnippetStats) {
	for _, snippet := range snippets {
		if pc.CheckIfSnippetIsPopular(snippet) {
			snippet.Badges = append(snippet.Badges, &aviaSearchProto.Badge{
				Type:          aviaSearchProto.BadgeType_BADGE_TYPE_POPULAR,
				OptionalValue: nil,
			})
		}
	}

	return snippets, cacheSnippetStats
}
