package search

import (
	"context"
	"fmt"
	"time"

	"github.com/go-redis/redis/v8"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	aviaApi "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/common"
)

const AviaAppRedisCacheVersion = "5"

type AviaAppRedisCacheConfig struct {
	// Сколько времени держим данные в кэше
	CacheTTL            time.Duration `config:"cache_ttl" yaml:"cache_ttl"`
	Environment         string
	Version             string
	AdditionalKeyPrefix string `config:"LOGIN"`
}

var DefaultAviaAppRedisCacheConfig = AviaAppRedisCacheConfig{
	CacheTTL: 20 * time.Minute,
	Version:  "5",
}

type AviaAppRedisCache struct {
	cfg           *AviaAppRedisCacheConfig
	env           string
	RedisClient   redis.UniversalClient
	logger        log.Logger
	serviceSearch *ServiceSearch
}

func NewAviaAppRedisCache(
	cfg *AviaAppRedisCacheConfig,
	env string,
	l log.Logger,
	redisClient redis.UniversalClient,
	serviceSearch *ServiceSearch,
) *AviaAppRedisCache {
	if env == string(common.DevelopmentEnv) {
		env = fmt.Sprintf("%s/%s", env, cfg.AdditionalKeyPrefix)
	}

	return &AviaAppRedisCache{
		cfg:           cfg,
		env:           env,
		RedisClient:   redisClient,
		logger:        l,
		serviceSearch: serviceSearch,
	}
}

type CacheResult struct {
	snippets []*aviaApi.Snippet
}

func (rc *AviaAppRedisCache) GenerateKey(qid string) string {
	return fmt.Sprintf("%s/%s/%s", rc.cfg.Environment, rc.cfg.Version, qid)
}

func (rc *AviaAppRedisCache) Get(ctx context.Context, qid string, sortBy aviaApi.SearchSort, filters *aviaApi.SearchFiltersReq) (*aviaSearchProto.SearchResult, error) {
	rawKey := rc.GenerateKey(qid)

	if rc.RedisClient == nil {
		return nil, nil
	}

	v, err := rc.RedisClient.Get(ctx, rawKey).Bytes()
	if err == redis.Nil {
		return nil, nil
	} else if err != nil {
		return nil, err
	}

	value := &aviaSearchProto.SearchResult{}
	err = proto.Unmarshal(v, value)
	if err != nil {
		return nil, fmt.Errorf("value deserialize error: %w", err)
	}

	return value, nil
}

func (rc *AviaAppRedisCache) Set(ctx context.Context, qid string, value *aviaSearchProto.SearchResult) (*aviaSearchProto.SearchResult, error) {
	realWrite := value.Progress.Current == value.Progress.Total

	key := rc.GenerateKey(qid)
	bytesValue, err := proto.Marshal(value)
	if err != nil {
		return value, err
	}

	if realWrite && rc.RedisClient != nil {
		if err := rc.RedisClient.Set(ctx, key, bytesValue, rc.cfg.CacheTTL).Err(); err != nil {
			return value, err
		}
	}

	return value, nil
}
