package search

import (
	"context"
	"strconv"
	"time"

	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	aviaApi "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	commonV1 "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/badges"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering2"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/sorting"
	"a.yandex-team.ru/travel/app/backend/internal/common"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviabackendclient"
	"a.yandex-team.ru/travel/app/backend/internal/lib/aviatdapiclient"
	"a.yandex-team.ru/travel/avia/library/go/searchcontext"
	"a.yandex-team.ru/travel/avia/library/go/services/featureflag"
	commonV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

type CacheFormatToolsConfig struct {
	BadgesObserver badges.CombinedBadgeObserverBuilderConfig
	// Через сколько времени протухает выдача в мобильном приложении
	ResultExpiration time.Duration `config:"result_expiration" yaml:"result_expiration"`
}

var DefaultCacheFormatToolsConfig = CacheFormatToolsConfig{
	BadgesObserver:   badges.DefaultCombinedBadgeObserverBuilderConfig,
	ResultExpiration: 15 * time.Minute,
}

type CacheFormatTools struct {
	cfg                  *CacheFormatToolsConfig
	cacheCfg             *AviaAppRedisCacheConfig
	serviceSearch        *ServiceSearch
	aviaBackendClient    BackendClient
	sorter               sorting.Sorter
	defaultSort          aviaApi.SearchSort
	logger               log.Logger
	badgeObserverBuilder badges.BadgeObserverBuilder
	featureFlag          featureflag.StorageInterface
}

func NewCacheFormatTools(
	cfg *CacheFormatToolsConfig,
	cacheCfg *AviaAppRedisCacheConfig,
	serviceSearch *ServiceSearch,
	aviaBackendClient BackendClient,
	featureFlag featureflag.StorageInterface,
	sorterCfg *sorting.ManagerSorterConfig,
	logger log.Logger,
) *CacheFormatTools {
	sorter := sorting.NewManagerSorter(sorterCfg, featureFlag)
	return &CacheFormatTools{
		cfg:                  cfg,
		cacheCfg:             cacheCfg,
		serviceSearch:        serviceSearch,
		aviaBackendClient:    aviaBackendClient,
		sorter:               sorter,
		defaultSort:          sorterCfg.DefaultSort,
		logger:               logger,
		badgeObserverBuilder: badges.NewCombinedBadgeObserverBuilder(&cfg.BadgesObserver, logger),
		featureFlag:          featureFlag,
	}
}

func BuildTransfersForResponse(transfers []*aviaSearchProto.Transfer) []*aviaApi.Snippet_Transfer {
	converted := make([]*aviaApi.Snippet_Transfer, 0, len(transfers))
	for _, elem := range transfers {
		converted = append(converted, &aviaApi.Snippet_Transfer{
			ArrivalStationId:   elem.ArrivalStationId,
			DepartureStationId: elem.DepartureStationId,
			NightTransfer:      elem.NightTransfer,
			AirportChange:      elem.AirportChange,
			ToTrain:            elem.ToTrain,
			ToBus:              elem.ToBus,
			SelfConnect:        elem.SelfConnect,
			DurationMinutes:    elem.DurationMinutes,
			DateChanged:        false,
		})
	}

	return converted
}

func GetConvertedBadges(badges []*aviaSearchProto.Badge) []*aviaApi.Snippet_Badge {
	converted := make([]*aviaApi.Snippet_Badge, 0)

	for _, badge := range badges {
		convertedBadge := &aviaApi.Snippet_Badge{
			Type: aviaApi.Snippet_BadgeType(badge.Type),
		}

		switch optionalValue := badge.OptionalValue.(type) {
		case *aviaSearchProto.Badge_PlusCashbackPoints:
			convertedBadge.OptionalValue = &aviaApi.Snippet_Badge_PlusCashbackPoints{
				PlusCashbackPoints: optionalValue.PlusCashbackPoints,
			}
		}

		converted = append(converted, convertedBadge)
	}

	return converted
}

func (cft *CacheFormatTools) BuildSnippetsForResponse(ctx context.Context, qid *searchcontext.QID, value *aviaSearchProto.SearchResult, sortBy aviaApi.SearchSort, filters *aviaApi.SearchFiltersReq, cacheSnippetStats *aviaSearchProto.CacheSnippetStats) ([]*aviaApi.Snippet, *aviaApi.SearchFiltersRsp) {
	badgesObserver := cft.badgeObserverBuilder.BuildToResponse(ctx, &badges.AviaClients{AviaBackendClient: cft.aviaBackendClient}, qid, sortBy, value.Reference, cacheSnippetStats)

	snippets, filtersResponse := cft.ApplyFilters(ctx, value.Snippets, value.Reference, value.SearchContext, filters)
	value.Snippets = snippets

	var bestPrice *commonV1.Price = nil
	var bestForward, bestBackward []string
	bestVariantSnippet := make(map[string]*aviaSearchProto.Variant)

	for _, snippet := range value.Snippets {
		bestVariantKey := ""
		var bestVariantPrice *commonV1.Price = nil
		for key, value := range snippet.Variant {
			valuePrice := &commonV1.Price{
				Currency: value.Price.Currency,
				Value:    value.Price.Value,
			}

			if bestVariantPrice != nil && searchcommon.CompareOnlyPriceAsc(bestVariantPrice, valuePrice) {
				continue
			}

			bestVariantKey = key
			bestVariantPrice = valuePrice
		}

		bestVariant, found := snippet.Variant[bestVariantKey]
		if !found {
			continue
		}
		bestVariantSnippet[snippet.Key] = bestVariant

		if bestPrice == nil || searchcommon.CompareExtendedPriceAsc(bestVariantPrice, bestPrice, snippet.Forward, bestForward, snippet.Backward, bestBackward, value.Reference.Flights) {
			bestPrice = bestVariantPrice
		}
	}

	snippetsStats := &badges.SnippetStats{BestPrice: bestPrice}

	converted := make([]*aviaApi.Snippet, 0, len(value.Snippets))
	for _, snippet := range value.Snippets {
		transfers := &aviaApi.Snippet_Transfers{
			ForwardTransfers:  BuildTransfersForResponse(snippet.Transfers.ForwardTransfers),
			BackwardTransfers: BuildTransfersForResponse(snippet.Transfers.BackwardTransfers),
		}

		bestVariant := bestVariantSnippet[snippet.Key]

		baggage := &aviaApi.Snippet_Baggage{
			Included: bestVariant.Baggage.Included,
		}
		if bestVariant.Baggage.OptionalWeight != nil && bestVariant.Baggage.OptionalWeight.(*aviaSearchProto.Baggage_Weight) != nil {
			baggage.OptionalWeight = &aviaApi.Snippet_Baggage_Weight{Weight: bestVariant.Baggage.GetWeight()}
		}
		if bestVariant.Baggage.OptionalPieces != nil && bestVariant.Baggage.OptionalPieces.(*aviaSearchProto.Baggage_Pieces) != nil {
			baggage.OptionalPieces = &aviaApi.Snippet_Baggage_Pieces{Pieces: bestVariant.Baggage.GetPieces()}
		}

		carryOn := &aviaApi.Snippet_CarryOn{
			Included: bestVariant.CarryOn.Included,
		}
		if bestVariant.CarryOn.OptionalWeight != nil && bestVariant.CarryOn.OptionalWeight.(*aviaSearchProto.CarryOn_Weight) != nil {
			carryOn.OptionalWeight = &aviaApi.Snippet_CarryOn_Weight{Weight: bestVariant.CarryOn.GetWeight()}
		}

		refund := &aviaApi.Snippet_Refund{
			Availability: aviaApi.Snippet_RefundAvailability(bestVariant.Refund.Availability),
		}
		if bestVariant.Refund.Charge != nil {
			refund.Price = &commonV1.Price{
				Currency: bestVariant.Refund.Charge.Currency,
				Value:    bestVariant.Refund.Charge.Value,
			}
		}

		convertedVariant := &aviaApi.Snippet_Variant{
			Key:         bestVariant.Key,
			PartnerCode: bestVariant.PartnerCode,
			Price: &commonV1.Price{
				Currency: bestVariant.Price.Currency,
				Value:    bestVariant.Price.Value,
			},
			Baggage:          baggage,
			CarryOn:          carryOn,
			Refund:           refund,
			OrderRelativeUrl: bestVariant.OrderRelativeUrl,
		}

		convertedSnippet := &aviaApi.Snippet{
			Key:                     snippet.Key,
			Forward:                 snippet.Forward,
			Backward:                snippet.Backward,
			Variant:                 convertedVariant,
			Badges:                  append(GetConvertedBadges(snippet.Badges), GetConvertedBadges(bestVariant.Badges)...),
			Transfers:               transfers,
			ForwardDurationMinutes:  snippet.ForwardDurationMinutes,
			BackwardDurationMinutes: snippet.BackwardDurationMinutes,
		}

		convertedSnippet = badgesObserver.ObserveSnippet(convertedSnippet, snippetsStats)
		converted = append(converted, convertedSnippet)
	}

	converted = cft.sorter.Sort(converted, value.Reference.Flights, sortBy)

	return converted, filtersResponse
}

func BuildReferenceForResponse(ff *aviaSearchProto.Reference) *aviaApi.SearchResultReference {
	flights := make(map[string]*aviaApi.SearchResultReference_Flight, len(ff.Flights))
	for key, value := range ff.Flights {
		flights[key] = &aviaApi.SearchResultReference_Flight{
			Key:             value.Key,
			AviaCompanyId:   value.AviaCompanyId,
			Number:          value.Number,
			StationFromId:   value.StationFromId,
			StationToId:     value.StationToId,
			Departure:       value.DepartureStr,
			Arrival:         value.ArrivalStr,
			DurationMinutes: value.DurationMinutes,
			DateChanged:     false,
		}
	}

	partners := make(map[string]*aviaApi.SearchResultReference_Partner, len(ff.Partners))
	for key, value := range ff.Partners {
		partners[key] = &aviaApi.SearchResultReference_Partner{
			Code:    value.Code,
			Title:   value.Title,
			LogoSvg: value.LogoSvg,
			LogoPng: value.LogoPng,
		}
	}

	settlements := make(map[uint64]*aviaApi.SearchResultReference_Settlement, len(ff.Settlements))
	for key, value := range ff.Settlements {
		settlements[key] = &aviaApi.SearchResultReference_Settlement{
			Id:               value.Id,
			Title:            value.Title,
			TitleGenitive:    value.TitleGenitive,
			TitleAccusative:  value.TitleAccusative,
			TitlePreposition: value.TitlePreposition,
			TitleLocative:    value.TitleLocative,
		}
	}

	stations := make(map[uint64]*aviaApi.SearchResultReference_Station, len(ff.Stations))
	for key, value := range ff.Stations {
		stations[key] = &aviaApi.SearchResultReference_Station{
			Id:               value.Id,
			AviaCode:         value.AviaCode,
			SettlementId:     value.SettlementId,
			Title:            value.Title,
			TitleGenitive:    value.TitleGenitive,
			TitleAccusative:  value.TitleAccusative,
			TitlePreposition: value.TitlePreposition,
			TitleLocative:    value.TitleLocative,
		}
	}

	aviaCompanies := make(map[uint64]*aviaApi.SearchResultReference_AviaCompany, len(ff.AviaCompanies))
	for key, value := range ff.AviaCompanies {
		aviaCompanies[key] = &aviaApi.SearchResultReference_AviaCompany{
			Id:         value.Id,
			AllianceId: value.AllianceId,
			Title:      value.Title,
			LogoSvg:    value.LogoSvg,
			LogoPng:    value.LogoPng,
			Color:      value.Color,
		}
	}

	alliances := make(map[uint64]*aviaApi.SearchResultReference_Alliance, len(ff.Alliances))
	for key, value := range ff.Alliances {
		alliances[key] = &aviaApi.SearchResultReference_Alliance{
			Id:    value.Id,
			Title: value.Title,
		}
	}

	return &aviaApi.SearchResultReference{
		Flights:       flights,
		Partners:      partners,
		Settlements:   settlements,
		Stations:      stations,
		AviaCompanies: aviaCompanies,
		Alliances:     alliances,
	}
}

func (cft *CacheFormatTools) BuildResponseFromCacheData(ctx context.Context, qid string, value *aviaSearchProto.SearchResult, sortBy aviaApi.SearchSort, filters *aviaApi.SearchFiltersReq, onlyFilters bool) (*aviaApi.SearchResultRsp, error) {
	parsedQid, err := searchcontext.ParseQID(qid)
	if err != nil {
		return nil, err
	}
	progress := value.Progress
	var snippets []*aviaApi.Snippet
	var builtFilters *aviaApi.SearchFiltersRsp
	if onlyFilters {
		_, builtFilters = cft.ApplyFilters(ctx, value.Snippets, value.Reference, value.SearchContext, filters)
	} else {
		snippets, builtFilters = cft.BuildSnippetsForResponse(ctx, &parsedQid, value, sortBy, filters, value.CacheSnippetStats)
	}

	if sortBy == aviaApi.SearchSort_SEARCH_SORT_UNKNOWN {
		sortBy = cft.defaultSort
	}
	return &aviaApi.SearchResultRsp{
		Qid:       qid,
		Reference: BuildReferenceForResponse(value.Reference),
		Snippets:  snippets,
		Progress: &aviaApi.SearchProgress{
			Current: progress.Current,
			Total:   progress.Total,
		},
		ExpiresAt:     common.FormatISO8601(time.Now().Add(cft.cfg.ResultExpiration)),
		SnippetsCount: uint32(len(snippets)),
		Sort:          sortBy,
		Filters:       builtFilters,
	}, nil
}

func (cft *CacheFormatTools) BuildTransfersForCache(transfers []*aviaApi.Snippet_Transfer) []*aviaSearchProto.Transfer {
	converted := make([]*aviaSearchProto.Transfer, 0)
	for _, t := range transfers {
		converted = append(converted, &aviaSearchProto.Transfer{
			ArrivalStationId:   t.ArrivalStationId,
			DepartureStationId: t.DepartureStationId,
			NightTransfer:      t.NightTransfer,
			AirportChange:      t.AirportChange,
			ToTrain:            t.ToTrain,
			ToBus:              t.ToBus,
			SelfConnect:        t.SelfConnect,
			DurationMinutes:    t.DurationMinutes,
		})
	}

	return converted
}

func (cft *CacheFormatTools) BuildSnippetsForCache(ctx context.Context, qid string, searchResultData *aviatdapiclient.SearchResultData) (map[string]*aviaSearchProto.Snippet, *aviaSearchProto.CacheSnippetStats, error) {
	cacheSnippetStats := &aviaSearchProto.CacheSnippetStats{}
	fares := searchResultData.Variants.Fares
	reference := searchResultData.Reference

	snippets := make(map[string]*aviaSearchProto.Snippet)

	parsedQid, err := searchcontext.ParseQID(qid)
	if err != nil {
		return nil, nil, err
	}

	badgesObserver := cft.badgeObserverBuilder.BuildToCache(ctx, &badges.AviaClients{AviaBackendClient: cft.aviaBackendClient}, &parsedQid, &searchResultData.Reference)

	flightReference := make(map[string]aviatdapiclient.Flight, len(reference.Flights))
	for _, f := range reference.Flights {
		flightReference[f.Key] = f
	}
	stationReference := make(map[uint64]aviatdapiclient.Station, len(reference.Stations))
	for _, st := range reference.Stations {
		stationReference[st.ID] = st
	}

	for _, fare := range fares {
		if len(fare.Prices) == 0 {
			continue
		}

		var transfers *aviaApi.Snippet_Transfers
		variants := make(map[string]*aviaSearchProto.Variant)

		for i, variantRaw := range fare.Prices {
			var err error
			transfers, err = BuildTransfers(fare.Route[0], fare.Route[1], variantRaw.Selfconnect, flightReference, stationReference)
			if err != nil {
				return nil, nil, xerrors.Errorf("build snippet transfers error: %w", err)
			}

			baggage := cft.serviceSearch.BuildBaggage(ctx, variantRaw.FareFamilies, reference.FareFamilies, variantRaw.Baggage, reference.BaggageTariffs)

			var price commonV1.Price
			if len(variantRaw.TariffNational.Currency) > 0 {
				price = commonV1.Price{
					Currency: variantRaw.TariffNational.GetCorrectedCurrency(),
					Value:    variantRaw.TariffNational.Value,
				}
			} else {
				price = commonV1.Price{
					Currency: variantRaw.Tariff.GetCorrectedCurrency(),
					Value:    variantRaw.Tariff.Value,
				}
			}

			refund := cft.serviceSearch.BuildRefund(ctx, variantRaw.FareFamilies, reference.FareFamilies)
			carryOn := cft.serviceSearch.BuildCarryOn(ctx, variantRaw.FareFamilies, reference.FareFamilies, fare.Route, reference.Flights, reference.CompanyTariffs)

			orderURL, err := BuildOrderURL(qid, fare.Route[0], fare.Route[1], flightReference, baggage.Included, refund.Availability)
			if err != nil {
				return nil, nil, xerrors.Errorf("error building order URL: %w", err)
			}

			key := strconv.Itoa(i)

			convertedRefund := &aviaSearchProto.Refund{
				Availability: aviaSearchProto.RefundAvailability(refund.Availability),
			}
			if refund.Price != nil {
				convertedRefund.Charge = &commonV2.Price{
					Currency: refund.Price.Currency,
					Value:    refund.Price.Value,
				}
			}

			var baggageOptionalWeight *aviaSearchProto.Baggage_Weight = nil
			if baggage.GetOptionalWeight() != nil {
				baggageOptionalWeight = &aviaSearchProto.Baggage_Weight{Weight: baggage.GetWeight()}
			}

			var baggageOptionalPieces *aviaSearchProto.Baggage_Pieces = nil
			if baggage.GetOptionalPieces() != nil {
				baggageOptionalPieces = &aviaSearchProto.Baggage_Pieces{Pieces: baggage.GetPieces()}
			}

			var carryOnWeight *aviaSearchProto.CarryOn_Weight = nil
			if carryOn.GetOptionalWeight() != nil {
				carryOnWeight = &aviaSearchProto.CarryOn_Weight{Weight: carryOn.GetWeight()}
			}

			variantBadges := badgesObserver.ObserveRawVariant(&variantRaw, nil)

			variants[key] = &aviaSearchProto.Variant{
				Key:         key,
				PartnerCode: variantRaw.PartnerCode,
				Price: &commonV2.Price{
					Currency: price.Currency,
					Value:    price.Value,
				},
				OrderRelativeUrl: orderURL,
				Baggage: &aviaSearchProto.Baggage{
					Included:       baggage.Included,
					OptionalWeight: baggageOptionalWeight,
					OptionalPieces: baggageOptionalPieces,
				},
				CarryOn: &aviaSearchProto.CarryOn{
					Included:       carryOn.Included,
					OptionalWeight: carryOnWeight,
				},
				Refund: convertedRefund,
				Badges: variantBadges,
			}

		}
		if len(variants) == 0 {
			continue
		}

		snippetKey, err := BuildSnippetKey(fare.Route)
		if err != nil {
			return nil, nil, xerrors.Errorf("build snippet key error: %w", err)
		}

		forwardDurationMinutes, err := CalcDurationMinutes(fare.Route[0], flightReference)
		if err != nil {
			return nil, nil, xerrors.Errorf("calc forward duration error: %w", err)
		}
		backwardDurationMinutes, err := CalcDurationMinutes(fare.Route[1], flightReference)
		if err != nil {
			return nil, nil, xerrors.Errorf("calc backward duration error: %w", err)
		}

		snippet := &aviaSearchProto.Snippet{
			Key:      snippetKey,
			Forward:  fare.Route[0],
			Backward: fare.Route[1],
			Transfers: &aviaSearchProto.Transfers{
				ForwardTransfers:  cft.BuildTransfersForCache(transfers.ForwardTransfers),
				BackwardTransfers: cft.BuildTransfersForCache(transfers.BackwardTransfers),
			},
			Variant:                 variants,
			Badges:                  nil,
			ForwardDurationMinutes:  forwardDurationMinutes,
			BackwardDurationMinutes: backwardDurationMinutes,
		}

		snippet = badgesObserver.ObserveSnippet(&fare, snippet)
		snippets[snippetKey] = snippet
	}

	snippets, cacheSnippetStats = badgesObserver.ObserveAllSnippets(snippets, cacheSnippetStats)
	return snippets, cacheSnippetStats, nil
}

func BuildReferenceForCache(ff *aviaApi.SearchResultReference) *aviaSearchProto.Reference {
	flights := make(map[string]*aviaSearchProto.Flight, len(ff.Flights))
	for key, value := range ff.Flights {
		depTime, _ := time.Parse(time.RFC3339, value.Departure)
		arrTime, _ := time.Parse(time.RFC3339, value.Arrival)

		flights[key] = &aviaSearchProto.Flight{
			Key:             value.Key,
			AviaCompanyId:   value.AviaCompanyId,
			Number:          value.Number,
			StationFromId:   value.StationFromId,
			StationToId:     value.StationToId,
			Departure:       timestamppb.New(depTime),
			DepartureStr:    value.Departure,
			Arrival:         timestamppb.New(arrTime),
			ArrivalStr:      value.Arrival,
			DurationMinutes: value.DurationMinutes,
		}
	}

	partners := make(map[string]*aviaSearchProto.Partner, len(ff.Partners))
	for key, value := range ff.Partners {
		partners[key] = &aviaSearchProto.Partner{
			Code:    value.Code,
			Title:   value.Title,
			LogoSvg: value.LogoSvg,
			LogoPng: value.LogoPng,
		}
	}

	settlements := make(map[uint64]*aviaSearchProto.Settlement, len(ff.Settlements))
	for key, value := range ff.Settlements {
		settlements[key] = &aviaSearchProto.Settlement{
			Id:               value.Id,
			Title:            value.Title,
			TitleGenitive:    value.TitleGenitive,
			TitleAccusative:  value.TitleAccusative,
			TitlePreposition: value.TitlePreposition,
			TitleLocative:    value.TitleLocative,
		}
	}

	stations := make(map[uint64]*aviaSearchProto.Station, len(ff.Stations))
	for key, value := range ff.Stations {
		stations[key] = &aviaSearchProto.Station{
			Id:               value.Id,
			AviaCode:         value.AviaCode,
			SettlementId:     value.SettlementId,
			Title:            value.Title,
			TitleGenitive:    value.TitleGenitive,
			TitleAccusative:  value.TitleAccusative,
			TitlePreposition: value.TitlePreposition,
			TitleLocative:    value.TitleLocative,
		}
	}

	aviaCompanies := make(map[uint64]*aviaSearchProto.AviaCompany, len(ff.AviaCompanies))
	for key, value := range ff.AviaCompanies {
		aviaCompanies[key] = &aviaSearchProto.AviaCompany{
			Id:         value.Id,
			AllianceId: value.AllianceId,
			Title:      value.Title,
			LogoSvg:    value.LogoSvg,
			LogoPng:    value.LogoPng,
			Color:      value.Color,
		}
	}

	alliances := make(map[uint64]*aviaSearchProto.Alliance, len(ff.Alliances))
	for key, value := range ff.Alliances {
		alliances[key] = &aviaSearchProto.Alliance{
			Id:    value.Id,
			Title: value.Title,
		}
	}

	return &aviaSearchProto.Reference{
		Flights:       flights,
		Partners:      partners,
		Settlements:   settlements,
		Stations:      stations,
		AviaCompanies: aviaCompanies,
		Alliances:     alliances,
	}
}

func (cft *CacheFormatTools) BuildCacheData(ctx context.Context, qid string, searchResultData aviatdapiclient.SearchResultData) (*aviaSearchProto.SearchResult, error) {
	searchContext, err := searchcommon.ParseQIDToProto(qid)
	if err != nil {
		return nil, err
	}

	aviaCompanies, err := BuildAviaCompaniesReference(searchResultData.Reference.Companies)
	if err != nil {
		return nil, err
	}

	flights, err := BuildFlightsReference(searchResultData.Reference.Flights)
	if err != nil {
		return nil, err
	}

	reference := BuildReferenceForCache(&aviaApi.SearchResultReference{
		Flights:       flights,
		Partners:      BuildPartnersReference(searchResultData.Reference.Partners),
		Settlements:   BuildSettlementsReference(searchResultData.Reference.Settlements),
		Stations:      BuildStationsReference(searchResultData.Reference.Stations),
		AviaCompanies: aviaCompanies,
		Alliances:     BuildAlliancesReference(searchResultData.Reference.Alliances),
	})

	snippets, snippetStats, err := cft.BuildSnippetsForCache(ctx, qid, &searchResultData)
	if err != nil {
		return nil, err
	}

	// TODO: TRAVELAPP-1018 - make all sorts
	sortedKeys := make([]*aviaSearchProto.VariantFullKey, 0, len(snippets))
	id := 0
	for key := range snippets {
		sortedKeys = append(sortedKeys, &aviaSearchProto.VariantFullKey{
			SnippentKey: key,
			VariantKey:  strconv.Itoa(id),
		})
		id += 1
	}

	value := &aviaSearchProto.SearchResult{
		Version:       cft.cacheCfg.Version,
		SearchContext: searchContext,
		Progress: &aviaSearchProto.Progress{
			Current: searchResultData.Progress.Current,
			Total:   searchResultData.Progress.Total,
		},
		Reference:                reference,
		Snippets:                 snippets,
		ExpiresAt:                timestamppb.New(time.Now().Add(cft.cacheCfg.CacheTTL)),
		SortedByRecommendedFirst: sortedKeys,
		SortedByCheapestFirst:    sortedKeys,
		SortedByExpensiveFirst:   sortedKeys,
		SortedByDeparture:        sortedKeys,
		SortedByArrival:          sortedKeys,
		CacheSnippetStats:        snippetStats,
	}

	return value, nil
}

func (cft *CacheFormatTools) ApplyFilters(
	ctx context.Context,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filters *aviaApi.SearchFiltersReq,
) (
	map[string]*aviaSearchProto.Snippet,
	*aviaApi.SearchFiltersRsp,
) {
	filtersDisableOnly := cft.featureFlag.GetFlags().IsFlagEnabledWithAB(ctx, common.AviaFiltersDisableOnlyFlag)
	ctxlog.Debugf(ctx, cft.logger, "filtersDisableOnly %t", filtersDisableOnly)
	if filtersDisableOnly {
		return filtering2.ApplyFilters(ctx, cft.logger, snippets, reference, searchContext, filters)
	} else {
		return filtering.ApplyFilters(ctx, cft.logger, snippets, reference, searchContext, filters)
	}
}

type BackendClient interface {
	TopFlights(ctx context.Context, nationalVersion, lang string, fromKey, toKey, date string, limit int) (*aviabackendclient.TopFlightsRsp, error)
}
