package filtering

import (
	"context"
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
)

type searchFilterInterface interface {
	getFilterID() string
	initFilterResponse(
		ctx context.Context,
		filters *aviaAPI.SearchFiltersReq,
		snippets map[string]*aviaSearchProto.Snippet,
		reference *aviaSearchProto.Reference,
		searchContext *aviaSearchProto.SearchContext,
		filterResponse *aviaAPI.SearchFiltersRsp,
	) *aviaAPI.SearchFiltersRsp
	filter(
		filters *aviaAPI.SearchFiltersReq,
		snippets map[string]*aviaSearchProto.Snippet,
		reference *aviaSearchProto.Reference,
		filterResponse *aviaAPI.SearchFiltersRsp,
	) (excludedSnippetKeys map[string]struct{}, excludedVariantKeys map[string]struct{})
	updateFilterResponse(
		ctx context.Context,
		snippets map[string]*aviaSearchProto.Snippet,
		excludedSnippetKeysByOthers map[string]struct{},
		excludedVariantKeysByOthers map[string]struct{},
		reference *aviaSearchProto.Reference,
		filterResponse *aviaAPI.SearchFiltersRsp,
	) *aviaAPI.SearchFiltersRsp
}

func ApplyFilters(
	ctx context.Context,
	logger log.Logger,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filters *aviaAPI.SearchFiltersReq,
) (
	map[string]*aviaSearchProto.Snippet,
	*aviaAPI.SearchFiltersRsp,
) {
	fs := []searchFilterInterface{
		&quickFilters{},
		newTransferFilter(filters),
		newAirportFilter(filters, snippets, reference, searchContext),
		newAviacompanyFilter(logger, filters),
		&timeFilter{},
	}

	filterResponse := &aviaAPI.SearchFiltersRsp{}
	for _, f := range fs {
		filterResponse = f.initFilterResponse(ctx, filters, snippets, reference, searchContext, filterResponse)
	}

	// Для понимания что дизейблить нужно знать, что осталось после других фильтров.
	// Поэтому сохраняем ключи того, что нужно отфильтровать, для каждого фильтра.
	excludedSnippetKeysByFilter := make(map[string]map[string]struct{}, len(fs))
	excludedVariantKeysByFilter := make(map[string]map[string]struct{}, len(fs))
	for _, f := range fs {
		excludedSnippetKeysByFilter[f.getFilterID()], excludedVariantKeysByFilter[f.getFilterID()] = f.filter(filters, snippets, reference, filterResponse)
	}

	// Для дизейбла, строим что отфильтровали другие фильтры.
	for _, f := range fs {
		excludedSnippetKeysByOthers := make(map[string]struct{})
		for fID, ek := range excludedSnippetKeysByFilter {
			if fID != f.getFilterID() {
				for k := range ek {
					excludedSnippetKeysByOthers[k] = struct{}{}
				}
			}
		}
		excludedVariantKeysByOthers := make(map[string]struct{})
		for fID, ek := range excludedVariantKeysByFilter {
			if fID != f.getFilterID() {
				for k := range ek {
					excludedVariantKeysByOthers[k] = struct{}{}
				}
			}
		}
		filterResponse = f.updateFilterResponse(
			ctx,
			snippets,
			excludedSnippetKeysByOthers,
			excludedVariantKeysByOthers,
			reference,
			filterResponse,
		)
	}

	// Фильтруем сниппеты и варианты.
	excludedSnippetKeys := make(map[string]struct{})
	for _, ek := range excludedSnippetKeysByFilter {
		for k := range ek {
			excludedSnippetKeys[k] = struct{}{}
		}
	}
	excludedVariantKeys := make(map[string]struct{})
	for _, ek := range excludedVariantKeysByFilter {
		for k := range ek {
			excludedVariantKeys[k] = struct{}{}
		}
	}
	refinedSnippets := make(map[string]*aviaSearchProto.Snippet)
	for sKey, s := range snippets {
		_, needToSkip := excludedSnippetKeys[sKey]
		if !needToSkip {
			variants := make(map[string]*aviaSearchProto.Variant, len(s.Variant))
			for vKey, v := range s.Variant {
				_, needToSkipVariant := excludedVariantKeys[helpers.BuildFullVariantKey(sKey, vKey)]
				if !needToSkipVariant {
					variants[vKey] = v
				}
			}
			refinedSnippets[sKey] = &aviaSearchProto.Snippet{
				Key:                     s.Key,
				Forward:                 s.Forward,
				Backward:                s.Backward,
				Transfers:               s.Transfers,
				Variant:                 variants,
				Badges:                  s.Badges,
				ForwardDurationMinutes:  s.ForwardDurationMinutes,
				BackwardDurationMinutes: s.BackwardDurationMinutes,
			}
		}
	}
	return refinedSnippets, filterResponse
}

func CheckPreconditions(filters *aviaAPI.SearchFiltersReq) error {
	if filters == nil || filters.Transfer == nil {
		return nil
	}
	if filters.QuickTransfer != filters.Transfer.NoTransfer {
		return fmt.Errorf("filters: quick_transfer must be the same as transfer.no_transfer")
	}

	return nil
}
