package helpers

import (
	"fmt"
	"unicode"

	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	aviaProtoV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

func BuildFullVariantKey(snippetKey, variantKey string) string {
	return snippetKey + "@" + variantKey
}

func CompareCyrillicFirst(left, right string) bool {
	if len(right) == 0 {
		return true
	}
	if len(left) == 0 {
		return false
	}

	leftCyrillic := isCyrillic([]rune(left)[0])
	rightCyrillic := isCyrillic([]rune(right)[0])
	if leftCyrillic == rightCyrillic {
		return left < right
	} else {
		return leftCyrillic
	}
}

func isCyrillic(r rune) bool {
	return unicode.Is(unicode.Cyrillic, r)
}

func BuildDepartureTitle(point *aviaProtoV2.Point, reference *aviaSearchProto.Reference) string {
	settlement := findSettlementForTitle(point, reference)
	if settlement == nil {
		return "Вылет"
	}
	return fmt.Sprintf("Вылет из %s", settlement.TitleGenitive)
}

func BuildArrivalTitle(point *aviaProtoV2.Point, reference *aviaSearchProto.Reference) string {
	settlement := findSettlementForTitle(point, reference)
	if settlement == nil {
		return "Прилёт"
	}
	return fmt.Sprintf("Прилёт %s %s", settlement.TitlePreposition, settlement.TitleAccusative)
}

func findSettlementForTitle(point *aviaProtoV2.Point, reference *aviaSearchProto.Reference) *aviaSearchProto.Settlement {
	if reference == nil || reference.Settlements == nil {
		return nil
	}

	if point.Type == aviaProtoV2.PointType_POINT_TYPE_SETTLEMENT {
		if settlement, found := reference.Settlements[point.Id]; found {
			return settlement
		}
	} else if point.Type == aviaProtoV2.PointType_POINT_TYPE_STATION {
		if station, found := reference.Stations[point.Id]; found {
			if station.SettlementId != 0 {
				if settlement, found := reference.Settlements[station.SettlementId]; found {
					return settlement
				}
			}
		}
	}
	return nil
}

type KeyCounter map[uint64]int

func NewKeyCounter() KeyCounter {
	return make(map[uint64]int)
}

func (c KeyCounter) Increment(key uint64) {
	if count, seen := c[key]; seen {
		c[key] = count + 1
	} else {
		c[key] = 1
	}
}

func (c KeyCounter) GetCount(key uint64) int {
	count, has := c[key]
	if !has {
		return 0
	} else {
		return count
	}
}
