package filtering

import (
	"context"

	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	commonAPI "a.yandex-team.ru/travel/app/backend/api/common/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/searchcommon"
)

type quickFilters struct{}

func (q *quickFilters) getFilterID() string {
	return "QuickFilter"
}

func (q *quickFilters) initFilterResponse(
	ctx context.Context,
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	quickBaggage := &aviaAPI.SearchFiltersRsp_QuickBaggageFilter{
		State: &aviaAPI.SearchFiltersRsp_BoolFilterState{
			Enabled: true,
			Value:   false,
		},
	}

	if filters != nil {
		quickBaggage.State.Value = filters.QuickBaggage
		quickBaggage.State.Enabled = true
	}

	if len(snippets) == 0 {
		quickBaggage.State.Value = false
		quickBaggage.State.Enabled = false
		filterResponse.QuickBaggage = quickBaggage
		return filterResponse
	}

	// quick baggage filter state
	hasBaggage := false
	hasNoBaggage := false
	for _, snippet := range snippets {
		for _, v := range snippet.Variant {
			if v.Baggage.Included {
				hasBaggage = true
			} else {
				hasNoBaggage = true
			}
			if hasBaggage && hasNoBaggage {
				break
			}
		}
		if hasBaggage && hasNoBaggage {
			break
		}
	}
	if hasNoBaggage && !hasBaggage {
		quickBaggage.State.Value = false
		quickBaggage.State.Enabled = false
	}
	if hasBaggage && !hasNoBaggage {
		quickBaggage.State.Value = true
		quickBaggage.State.Enabled = false
	}

	filterResponse.QuickBaggage = quickBaggage
	return filterResponse
}

func (q *quickFilters) needToSkip(filters *aviaAPI.SearchFiltersRsp, variant *aviaSearchProto.Variant) bool {
	if filters.QuickBaggage.State.Value && !variant.Baggage.Included {
		return true
	}
	return false
}

func (q *quickFilters) filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) (map[string]struct{}, map[string]struct{}) {
	excludedSnippetKeys := make(map[string]struct{}, len(snippets))
	excludedVariantKeys := make(map[string]struct{})
	for sKey, s := range snippets {
		hasRefinedVariants := false
		excludedVariantKeysLocal := make(map[string]struct{}, len(s.Variant))
		for vKey, variant := range s.Variant {
			if q.needToSkip(filterResponse, variant) {
				excludedVariantKeysLocal[vKey] = struct{}{}
			} else {
				hasRefinedVariants = true
			}
		}
		if !hasRefinedVariants {
			excludedSnippetKeys[sKey] = struct{}{}
		} else {
			for vKey := range excludedVariantKeysLocal {
				excludedVariantKeys[helpers.BuildFullVariantKey(sKey, vKey)] = struct{}{}
			}
		}
	}
	return excludedSnippetKeys, excludedVariantKeys
}

func (q *quickFilters) updateFilterResponse(
	ctx context.Context,
	snippets map[string]*aviaSearchProto.Snippet,
	excludedSnippetKeysByOthers map[string]struct{},
	excludedVariantKeysByOthers map[string]struct{},
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	var withBaggageMinPrice *commonAPI.Price = nil
	var noBaggageMinPrice *commonAPI.Price = nil
	for sKey, s := range snippets {
		if _, needToSkip := excludedSnippetKeysByOthers[sKey]; needToSkip {
			continue
		}
		for vKey, v := range s.Variant {
			if _, needToSkipVariant := excludedVariantKeysByOthers[helpers.BuildFullVariantKey(sKey, vKey)]; needToSkipVariant {
				continue
			}
			price := &commonAPI.Price{
				Currency: v.Price.Currency,
				Value:    v.Price.Value,
			}
			if v.Baggage.Included {
				if withBaggageMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, withBaggageMinPrice) {
					withBaggageMinPrice = price
				}
			} else {
				if noBaggageMinPrice == nil || searchcommon.CompareOnlyPriceAsc(price, noBaggageMinPrice) {
					noBaggageMinPrice = price
				}
			}
		}
	}
	filterResponse.QuickBaggage.MinPriceNoBaggage = noBaggageMinPrice
	filterResponse.QuickBaggage.MinPriceWithBaggage = withBaggageMinPrice

	hasBaggage := withBaggageMinPrice != nil
	hasNoBaggage := noBaggageMinPrice != nil
	if hasNoBaggage && !hasBaggage {
		filterResponse.QuickBaggage.State.Value = false
		filterResponse.QuickBaggage.State.Enabled = false
	}
	if hasBaggage && !hasNoBaggage {
		filterResponse.QuickBaggage.State.Value = true
		filterResponse.QuickBaggage.State.Enabled = false
	}

	return filterResponse
}
