package filtering

import (
	"context"

	aviaAPI "a.yandex-team.ru/travel/app/backend/api/avia/v1"
	"a.yandex-team.ru/travel/app/backend/internal/avia/search/filtering/helpers"
	aviaSearchProto "a.yandex-team.ru/travel/app/backend/internal/avia/search/proto/v1"
	aviaProtoV2 "a.yandex-team.ru/travel/avia/library/proto/common/v2"
)

type timeFilter struct {
	needToApplyFilter bool
	roundTrip         bool
}

func (t *timeFilter) getFilterID() string {
	return "TimeFilter"
}

func (t *timeFilter) initFilterResponse(
	ctx context.Context,
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	searchContext *aviaSearchProto.SearchContext,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	t.roundTrip = searchContext.DateBackward != nil

	// Создаем объекты
	filterResponse.DepartureAndArrival = &aviaAPI.SearchFiltersRsp_DepartureAndArrivalFilter{
		ForwardDeparture: &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
			Title: helpers.BuildDepartureTitle(searchContext.PointFrom, reference),
		},
		ForwardArrival: &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
			Title: helpers.BuildArrivalTitle(searchContext.PointTo, reference),
		},
	}
	if t.roundTrip {
		filterResponse.DepartureAndArrival.BackwardDeparture = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
			Title: helpers.BuildDepartureTitle(searchContext.PointTo, reference),
		}
		filterResponse.DepartureAndArrival.BackwardArrival = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalState{
			Title: helpers.BuildArrivalTitle(searchContext.PointFrom, reference),
		}
	}

	// Заполняем пункты отправления и прибытия
	if searchContext.PointFrom.Type == aviaProtoV2.PointType_POINT_TYPE_SETTLEMENT {
		filterResponse.DepartureAndArrival.ForwardDeparture.SettlementId = searchContext.PointFrom.Id
		if t.roundTrip {
			filterResponse.DepartureAndArrival.BackwardArrival.SettlementId = searchContext.PointFrom.Id
		}
	} else if searchContext.PointFrom.Type == aviaProtoV2.PointType_POINT_TYPE_STATION {
		filterResponse.DepartureAndArrival.ForwardDeparture.StationId = searchContext.PointFrom.Id
		if t.roundTrip {
			filterResponse.DepartureAndArrival.BackwardArrival.StationId = searchContext.PointFrom.Id
		}
	}
	if searchContext.PointTo.Type == aviaProtoV2.PointType_POINT_TYPE_SETTLEMENT {
		filterResponse.DepartureAndArrival.ForwardArrival.SettlementId = searchContext.PointTo.Id
		if t.roundTrip {
			filterResponse.DepartureAndArrival.BackwardDeparture.SettlementId = searchContext.PointTo.Id
		}
	} else if searchContext.PointFrom.Type == aviaProtoV2.PointType_POINT_TYPE_STATION {
		filterResponse.DepartureAndArrival.ForwardArrival.StationId = searchContext.PointTo.Id
		if t.roundTrip {
			filterResponse.DepartureAndArrival.BackwardDeparture.StationId = searchContext.PointTo.Id
		}
	}

	if len(snippets) == 0 {
		t.needToApplyFilter = false
		return filterResponse
	}

	// Проставляем максимальный интервал
	minForwardDeparture := ""
	maxForwardDeparture := ""
	minForwardArrival := ""
	maxForwardArrival := ""
	minBackwardDeparture := ""
	maxBackwardDeparture := ""
	minBackwardArrival := ""
	maxBackwardArrival := ""
	for _, s := range snippets {
		forwardDeparture := reference.Flights[s.Forward[0]].DepartureStr
		if minForwardDeparture == "" || minForwardDeparture > forwardDeparture {
			minForwardDeparture = forwardDeparture
		}
		if maxForwardDeparture == "" || maxForwardDeparture < forwardDeparture {
			maxForwardDeparture = forwardDeparture
		}

		forwardArrival := reference.Flights[s.Forward[len(s.Forward)-1]].ArrivalStr
		if minForwardArrival == "" || minForwardArrival > forwardArrival {
			minForwardArrival = forwardArrival
		}
		if maxForwardArrival == "" || maxForwardArrival < forwardArrival {
			maxForwardArrival = forwardArrival
		}

		if t.roundTrip {
			backwardDeparture := reference.Flights[s.Backward[0]].DepartureStr
			if minBackwardDeparture == "" || minBackwardDeparture > backwardDeparture {
				minBackwardDeparture = backwardDeparture
			}
			if maxBackwardDeparture == "" || maxBackwardDeparture < backwardDeparture {
				maxBackwardDeparture = backwardDeparture
			}

			backwardArrival := reference.Flights[s.Backward[len(s.Backward)-1]].ArrivalStr
			if minBackwardArrival == "" || minBackwardArrival > backwardArrival {
				minBackwardArrival = backwardArrival
			}
			if maxBackwardArrival == "" || maxBackwardArrival < backwardArrival {
				maxBackwardArrival = backwardArrival
			}
		}
	}
	filterResponse.DepartureAndArrival.ForwardDeparture.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: minForwardDeparture,
		MaxDatetime: maxForwardDeparture,
	}
	filterResponse.DepartureAndArrival.ForwardArrival.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: minForwardArrival,
		MaxDatetime: maxForwardArrival,
	}
	if t.roundTrip {
		filterResponse.DepartureAndArrival.BackwardDeparture.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
			MinDatetime: minBackwardDeparture,
			MaxDatetime: maxBackwardDeparture,
		}
		filterResponse.DepartureAndArrival.BackwardArrival.All = &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
			MinDatetime: minBackwardArrival,
			MaxDatetime: maxBackwardArrival,
		}
	}

	// Проставим выбранный интервал
	if filters != nil && filters.DepartureAndArrival != nil {
		f := filters.DepartureAndArrival
		filterResponse.DepartureAndArrival.ForwardDeparture.Selected = copyIntervalFromRequest(f.ForwardDeparture)
		filterResponse.DepartureAndArrival.ForwardArrival.Selected = copyIntervalFromRequest(f.ForwardArrival)
		if t.roundTrip {
			filterResponse.DepartureAndArrival.BackwardDeparture.Selected = copyIntervalFromRequest(f.BackwardDeparture)
			filterResponse.DepartureAndArrival.BackwardArrival.Selected = copyIntervalFromRequest(f.BackwardArrival)
		}
	} else {
		f := filterResponse.DepartureAndArrival
		f.ForwardDeparture.Selected = copyIntervalFromResponse(f.ForwardDeparture.All)
		f.ForwardArrival.Selected = copyIntervalFromResponse(f.ForwardArrival.All)
		if t.roundTrip {
			f.BackwardDeparture.Selected = copyIntervalFromResponse(f.BackwardDeparture.All)
			f.BackwardArrival.Selected = copyIntervalFromResponse(f.BackwardArrival.All)
		}
	}

	// Выбранное должно быть не больше максимального
	fitSelectedIntoAll(filterResponse.DepartureAndArrival.ForwardDeparture)
	fitSelectedIntoAll(filterResponse.DepartureAndArrival.ForwardArrival)
	if t.roundTrip {
		fitSelectedIntoAll(filterResponse.DepartureAndArrival.BackwardDeparture)
		fitSelectedIntoAll(filterResponse.DepartureAndArrival.BackwardArrival)
	}

	t.needToApplyFilter = needToApplyFilter(filterResponse.DepartureAndArrival.ForwardDeparture) ||
		needToApplyFilter(filterResponse.DepartureAndArrival.ForwardArrival)
	if t.roundTrip {
		t.needToApplyFilter = t.needToApplyFilter ||
			needToApplyFilter(filterResponse.DepartureAndArrival.BackwardDeparture) ||
			needToApplyFilter(filterResponse.DepartureAndArrival.BackwardArrival)
	}

	return filterResponse
}

func (t *timeFilter) filter(
	filters *aviaAPI.SearchFiltersReq,
	snippets map[string]*aviaSearchProto.Snippet,
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) (map[string]struct{}, map[string]struct{}) {
	if !t.needToApplyFilter {
		return nil, nil
	}

	excludedSnippetKeys := make(map[string]struct{}, len(snippets))
	for sKey, s := range snippets {
		if t.needToSkipSnippet(s, reference, filterResponse) {
			excludedSnippetKeys[sKey] = struct{}{}
		}
	}
	return excludedSnippetKeys, nil
}

func (t *timeFilter) needToSkipSnippet(s *aviaSearchProto.Snippet, reference *aviaSearchProto.Reference, filterResponse *aviaAPI.SearchFiltersRsp) bool {
	f := filterResponse.DepartureAndArrival
	if !inInterval(reference.Flights[s.Forward[0]].DepartureStr, f.ForwardDeparture.Selected) {
		return true
	}
	if !inInterval(reference.Flights[s.Forward[len(s.Forward)-1]].ArrivalStr, f.ForwardArrival.Selected) {
		return true
	}
	if t.roundTrip {
		if !inInterval(reference.Flights[s.Backward[0]].DepartureStr, f.BackwardDeparture.Selected) {
			return true
		}
		if !inInterval(reference.Flights[s.Backward[len(s.Backward)-1]].ArrivalStr, f.BackwardArrival.Selected) {
			return true
		}
	}
	return false
}

func inInterval(value string, interval *aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval) bool {
	if interval == nil {
		return true
	}
	return interval.MinDatetime <= value && value <= interval.MaxDatetime
}

func (t *timeFilter) updateFilterResponse(
	ctx context.Context,
	snippets map[string]*aviaSearchProto.Snippet,
	excludedSnippetKeysByOthers map[string]struct{},
	excludedVariantKeysByOthers map[string]struct{},
	reference *aviaSearchProto.Reference,
	filterResponse *aviaAPI.SearchFiltersRsp,
) *aviaAPI.SearchFiltersRsp {
	return filterResponse
}

func copyIntervalFromRequest(from *aviaAPI.SearchFiltersReq_DepartureOrArrivalInterval) *aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval {
	if from == nil {
		return nil
	}
	return &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: from.MinDatetime,
		MaxDatetime: from.MaxDatetime,
	}
}

func copyIntervalFromResponse(from *aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval) *aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval {
	if from == nil {
		return nil
	}
	return &aviaAPI.SearchFiltersRsp_DepartureOrArrivalInterval{
		MinDatetime: from.MinDatetime,
		MaxDatetime: from.MaxDatetime,
	}
}

func fitSelectedIntoAll(v *aviaAPI.SearchFiltersRsp_DepartureOrArrivalState) {
	if v != nil {
		if v.All != nil {
			if v.Selected == nil {
				v.Selected = copyIntervalFromResponse(v.All)
			} else {
				if v.Selected.MinDatetime == "" || v.Selected.MinDatetime < v.All.MinDatetime {
					v.Selected.MinDatetime = v.All.MinDatetime
				}
				if v.Selected.MaxDatetime == "" || v.Selected.MaxDatetime > v.All.MaxDatetime {
					v.Selected.MaxDatetime = v.All.MaxDatetime
				}
			}
		}
	}
}

func needToApplyFilter(v *aviaAPI.SearchFiltersRsp_DepartureOrArrivalState) bool {
	if v == nil {
		return false
	}
	if v.Selected == nil || v.All == nil {
		return false
	}
	return v.Selected != v.All
}
